<?php

/**
 * @package     EasyStore.Site
 * @subpackage  EasyStore.Stripe
 *
 * @copyright   Copyright (C) 2023 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Plugin\EasyStore\Stripe\Utils;

use Joomla\CMS\Factory;
use Stripe\StripeClient;
use Stripe\Checkout\Session;
use Stripe\Exception\ApiErrorException;



// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


/**
 * Class that contains Api for the Stripe payment gateway.
 * @since 1.0.0
*/
class StripeApi 
{
    /**
     * Get the checkout URL for Stripe.
     *
     * @param array  $lineItems      Line items for the checkout session.
     * @param string $orderID        The order ID.
     * @param object $productsList   The list of products and additional information.
     *
     * @return string               The checkout session URL.
     * @since  1.0.0
     */
    public static function getCheckoutUrl($lineItems,$orderID,$productsList) {

        $couponId        = '';
        $stripeConstants = new StripeConstants();

        $shippingOptions = StripeHelper::getShippingOptions($productsList);

        if ($productsList->coupon_discount_amount > 0) {   
            $coupon   =  self::createCoupon($productsList->coupon_discount_amount,$productsList->currency);
            $couponId = ($coupon->id) ? $coupon->id : '';
        }

        $checkout_session_data = [
            'line_items'    => $lineItems, 
            'metadata'      => ['payment_type'  => 'stripe', 'order_id' => $orderID , 'coupon_id' => $couponId],
            'mode'          => 'payment',
            'success_url'   => $stripeConstants->getSuccessUrl(),
            'cancel_url'    => $stripeConstants->getCancelUrl(),
            'payment_intent_data' => [
                'metadata' => ['order_id' => $orderID,'coupon_id' => $couponId],
            ],
            'shipping_options' => $shippingOptions,
        ];

        if ($couponId) {
            $checkout_session_data['discounts'] = [
                [
                    'coupon' => $couponId,
                ],
            ];
        }

       $checkout_session = Session::create($checkout_session_data);
        return $checkout_session->url;
    }


    /**
     * Create a coupon with a specific discount amount and currency.
     *
     * @param float  $couponAmount The discount amount to apply.
     * @param string $currency     The currency code.
     *
     * @return \Stripe\Coupon|null The created coupon object or null if an error occurred.
     * @since  1.0.0
     */
    public static function createCoupon($couponAmount,$currency){
        
        $stripeConstants = new StripeConstants();
        $stripeSecretKey = $stripeConstants->getStripeSecretKey();

        $stripe = new StripeClient($stripeSecretKey);

        try {
            $coupon = $stripe->coupons->create([
                        'amount_off' => $couponAmount * 100,
                        'duration'   => 'once',
                        'currency'   => strtolower($currency)
                    ]); 

            return $coupon;
        } catch (\Throwable $e) {
            Factory::getApplication()->enqueueMessage($e->getMessage(),'error');
        }
        
    }

    /**
     * Create a new webhook URL in Stripe with the given secret key.
     *
     * @param  string $stripeSecretKey - The Stripe secret key to use for creating the webhook.
     *
     * @return mixed                   - The newly created webhook object or an error message if an error occurs.
     * @since  1.0.0
     */
    public static function createNewWebhookUrl($stripeSecretKey){
        
        $stripe         = new StripeClient($stripeSecretKey);
        $stripeConstant = new StripeConstants();
        
        $webhookUrl = $stripeConstant->getWebHookUrl();

        try {
            $newCreatedWebhook = $stripe->webhookEndpoints->create([
                'url' => $webhookUrl,
                'enabled_events' => [
                    'payment_intent.payment_failed',
                    'checkout.session.completed',
                    'payment_intent.partially_funded',
                    'payment_intent.canceled'
                ],
            ]);

            return $newCreatedWebhook;
        } catch (ApiErrorException $error) {
            Factory::getApplication()->enqueueMessage($error->getMessage(), 'error');
        }
    }


    /**
     * Delete a coupon with the specified ID from Stripe.
     *
     * @param string $couponID The ID of the coupon to be deleted.
     * @since 1.0.0
     */
    public static function deleteCoupon($couponID){

        $stripeConstant  = new StripeConstants();
        $stripeSecretKey = $stripeConstant->getStripeSecretKey();
        
        $stripe = new StripeClient($stripeSecretKey);

        try {
            $stripe->coupons->delete($couponID, []);
        } catch (ApiErrorException $error) {
            Factory::getApplication()->enqueueMessage($error->getMessage(),'error');
        }
        
    }
} 