document.addEventListener('alpine:init', () => {
	Alpine.data('price_range', (options) => {
		return {
			init() {
				this.$watch('lowerThumbDragStart', (current) => {
					if (current) {
						this.igniteDragging();
					} else {
						this.takeOffDragging();
					}
				});
				this.$watch('upperThumbDragStart', (current) => {
					if (current) {
						this.igniteDragging();
					} else {
						this.takeOffDragging();
					}
				});

				this.$watch('lowerValue', (value) => {
					this.lowerThumbPercentValue = this.mapRange({
						value: value,
						valueMin: options.min,
						valueMax: options.max,
						toMin: 0,
						toMax: 100,
					});
				});

				this.$watch('upperValue', (value) => {
					this.upperThumbPercentValue = this.mapRange({
						value: value,
						valueMin: options.min,
						valueMax: options.max,
						toMin: 0,
						toMax: 100,
					});
				});

				this.lowerThumbPercentValue = this.mapRange({
					value: options.min_value,
					valueMin: options.min,
					valueMax: options.max,
					toMin: 0,
					toMax: 100,
				});

				this.upperThumbPercentValue = this.mapRange({
					value: options.max_value,
					valueMin: options.min,
					valueMax: options.max,
					toMin: 0,
					toMax: 100,
				});

				this.minDistance = Math.max(
					this.minDistance,
					this.mapRange({
						value: options.distance,
						valueMin: options.min,
						valueMax: options.max,
						toMin: 0,
						toMax: 100,
					}),
				);
			},

			options: options,
			lowerThumbDragStart: false,
			upperThumbDragStart: false,
			showLowerThumbTooltip: false,
			showUpperThumbTooltip: false,
			lowerThumbPercentValue: 0,
			upperThumbPercentValue: 100,
			minDistance: 16,
			lowerValue: options.min_value,
			upperValue: options.max_value,

			get distanceValue() {
				return this.mapRange({
					value: 1,
					toMin: options.min,
					toMax: options.max,
					valueMin: 0,
					valueMax: 100,
				});
			},

			get lowerThumbValue() {
				if (!this.lowerThumbDragStart) {
					return this.lowerValue;
				}
				this.lowerValue = this.mapRange({
					value: this.lowerThumbPercentValue,
					valueMin: 0,
					valueMax: 100,
					toMin: options.min,
					toMax: options.max,
				});

				return this.lowerValue;
			},

			get upperThumbValue() {
				if (!this.upperThumbDragStart) {
					return this.upperValue;
				}
				this.upperValue = this.mapRange({
					value: this.upperThumbPercentValue,
					valueMin: 0,
					valueMax: 100,
					toMin: options.min,
					toMax: options.max,
				});

				return this.upperValue;
			},

			updateValue(event, key) {
				const value = Number(event.value ?? '');

				if (key === 'lower') {
					if (value >= this.upperValue - this.distanceValue) {
						this.lowerValue = this.upperValue - this.distanceValue;
					} else {
						this.lowerValue = value;
					}

					this.lowerValue = Math.max(options.min, this.lowerValue);
				} else {
					if (value <= this.lowerValue + this.distanceValue) {
						this.upperValue = this.lowerValue + this.distanceValue;
					} else {
						this.upperValue = value;
					}
					this.upperValue = Math.min(options.max, this.upperValue);
				}

				this.updateUrlSearchParams();
			},

			igniteDragging() {
				window.addEventListener('mousemove', this.handleMouseMove.bind(this));
				window.addEventListener('mouseup', this.handleMouseEnd.bind(this));
				window.addEventListener('touchmove', this.handleMouseMove.bind(this));
				window.addEventListener('touchend', this.handleMouseEnd.bind(this));
			},

			takeOffDragging() {
				window.removeEventListener('mousemove', this.handleMouseMove.bind(this));
				window.removeEventListener('mouseup', this.handleMouseEnd.bind(this));
				window.removeEventListener('touchmove', this.handleMouseMove.bind(this));
				window.removeEventListener('touchend', this.handleMouseEnd.bind(this));
			},

			getLeftPosition(clientX) {
				const rangeSlider = this.$refs?.rangeSlider;
				if (!rangeSlider) {
					return 0;
				}

				const rect = rangeSlider.getBoundingClientRect();

				const width = Math.floor(rect.width);
				const clickOffset = Math.max(0, clientX - rect.left);
				const percent = Math.min(100, (clickOffset * 100) / width);

				return Math.round(percent);
			},

			handleMouseMove(event) {
				if (!this.lowerThumbDragStart && !this.upperThumbDragStart) {
					return;
				}

				document.body.style.userSelect = 'none';
				const clientX = event.touches ? event.touches[0]?.clientX : event.clientX;

				if (this.lowerThumbDragStart) {
					this.lowerThumbPercentValue = Math.min(
						this.upperThumbPercentValue - this.minDistance,
						this.getLeftPosition(clientX),
					);
					return;
				}

				if (this.upperThumbDragStart) {
					this.upperThumbPercentValue = Math.max(
						this.lowerThumbPercentValue + this.minDistance,
						this.getLeftPosition(clientX),
					);
					return;
				}
			},
			handleMouseEnd() {
				this.lowerThumbDragStart = false;
				this.upperThumbDragStart = false;
				document.body.style.userSelect = 'auto';
				this.updateUrlSearchParams();
			},

			mapRange({ value, valueMin, valueMax, toMin, toMax }) {
				const clampedValue = Math.min(Math.max(value, valueMin), valueMax);

				return Math.round(((clampedValue - valueMin) / (valueMax - valueMin)) * (toMax - toMin) + toMin);
			},

			updateUrlSearchParams() {
				const lowerValue = this.lowerThumbValue;
				const upperValue = this.upperThumbValue;

				const searchParams = new URLSearchParams(window.location.search);

				if (
					Number(searchParams.get('filter_min_price')) !== lowerValue ||
					Number(searchParams.get('filter_max_price')) !== upperValue
				) {
					searchParams.set('filter_min_price', lowerValue);
					searchParams.set('filter_max_price', upperValue);

					window.location.href = `${window.location.pathname}?${searchParams.toString()}`;
				}
			},
		};
	});
});
