document.addEventListener('alpine:init', () => {
	Alpine.data('easystore_product_review', (initialData) => ({
		data: {
			title: '',
			ratings: '',
			message: '',
			requiredValues: ['title', 'ratings'],
		},
		submitReview() {
			const isValid = this.data.requiredValues.every((field) => `${this.data[field]}`.trim());
			if (!isValid) return;

			const formData = new FormData();

			formData.append('easystore-title', this.data.title);
			formData.append('easystore-ratings', this.data.ratings);
			formData.append('easystore-message', this.data.message);
			formData.append('productId', initialData.productId);

			const url = `${Joomla.getOptions('easystore.base')}/index.php?option=com_easystore&task=product.addReview`;
			Joomla.request({
				url,
				method: 'POST',
				data: formData,
				onSuccess: (response) => {
					response = JSON.parse(response);
					if (response.status === 'success') {
						document.location.reload();
					}
				},
				onError: (xhr) => {
					Joomla.renderMessages(Joomla.ajaxErrorsMessages(xhr));
				},
			});
		},
	}));

	Alpine.directive('log', (el, { expression }, { evaluate }) => {
		console.log(evaluate(expression));
	});

	Alpine.data('easystore_invoice', () => ({
		easystorePrintInvoice() {
			window.print();
		},
	}));
});

document.addEventListener('alpine:init', () => {
	Alpine.data('easystoreProductDetails', () => {
		return {
			init() {
				this.setInitialActiveSource();

				this.$watch('openPreview', (value) => this.watchPreviewMode(value));
				this.$watch('zoomNextLevel', (value) => this.watchNextLevelZooming(value));
				this.$watch('currentSelectedIndex', (value) => this.watchSelectedImage(value));

				this.listenKeyboardEvent();
				this.controlVideoBasedOnDocumentVisibility();
			},
			getRef(ref) {
				const reference = this.$refs[ref] ?? document.querySelector(`[x-ref="${ref}"]`);

				if (!reference) {
					return null;
				}

				return reference;
			},

			quantity: 1,
			openPreview: false,
			zoomNextLevel: false,
			currentSelectedIndex: 0,
			activeSource: null,
			playState: false,
			activeMediaType: null,

			controlVideoBasedOnDocumentVisibility() {
				document.addEventListener('visibilitychange', () => {
					const video = this.getRef('previewVideo');
					if (!video) {
						return;
					}

					if (document.visibilityState === 'hidden') {
						this.pauseVideo(video);
					}

					if (document.visibilityState === 'visible') {
						this.playVideo(video);
					}
				});
			},

			listenKeyboardEvent() {
				window.addEventListener('keydown', (event) => {
					if (!this.openPreview) {
						return;
					}

					switch (event.key) {
						case 'Escape':
							this.openPreview = false;
							this.getRef('previewVideo').pause();
							break;
						case 'ArrowLeft':
							this.currentSelectedIndex = this.currentSelectedIndex - 1;
							if (this.currentSelectedIndex < 0) {
								this.currentSelectedIndex = this.totalImages - 1;
							}
							break;
						case 'ArrowRight':
							this.currentSelectedIndex = this.currentSelectedIndex + 1;
							if (this.currentSelectedIndex >= this.totalImages) {
								this.currentSelectedIndex = 0;
							}
							break;
						case ' ': {
							this.zoomNextLevel = !this.zoomNextLevel;

							if (this.activeMediaType !== 'video') {
								return;
							}

							this.playState = !this.playState;
							const video = this.getRef('previewVideo');

							if (this.playState) {
								this.playVideo(video);
							} else {
								this.pauseVideo(video);
							}

							break;
						}
					}
				});
			},

			toggleZoom() {
				this.zoomNextLevel = !this.zoomNextLevel;
			},
			onThumbnailChange(event, src, type) {
				let thumbElement = null;
				const productVideo = this.getRef('productVideo');
				const productThumbnail = this.getRef('productThumbnail');

				if (type === 'video') {
					thumbElement = productVideo.querySelector('video');
					productThumbnail.style.display = 'none';
					productVideo.style.display = 'block';
				} else {
					thumbElement = productThumbnail;
					productVideo.style.display = 'none';
					productThumbnail.style.display = 'block';
				}

				if (!thumbElement) {
					return;
				}

				this.activeSource = src;
				this.setCurrentSelectedIndex();
				thumbElement.src = src;
				const parent = event.currentTarget.parentNode;
				const buttons = parent.querySelectorAll('button');
				buttons.forEach((button) => button.classList.remove('active'));
				event.currentTarget.classList.add('active');
			},

			increment() {
				this.quantity = Number(this.quantity) + 1;
			},
			decrement() {
				this.quantity = Math.max(1, Number(this.quantity) - 1);
			},
			closeOpenedModalIfAny() {
				const modal = document.querySelector('#easystore-modal');
				modal.classList.remove('active');
				modal.setAttribute('aria-hidden', 'true');
				modal.setAttribute('aria-busy', 'false');
				document.body.classList.remove('easystore-modal-open');
			},
			addToCart(productId, skuId) {
				const formData = new FormData();
				const items = [
					{
						product_id: productId,
						sku_id: skuId || null,
						quantity: this.quantity,
					},
				];

				const url = `${Joomla.getOptions('easystore.base')}/index.php?option=com_easystore&task=cart.addToCart`;

				formData.append('items', JSON.stringify(items));
				formData.append('initiator', 'cart-button');
				formData.append('_method', 'POST');

				this.$el.classList.add('easystore-spinner');

				Joomla.request({
					url,
					method: 'POST',
					data: formData,
					onSuccess: () => {
						this.$el.classList.remove('easystore-spinner');
						this.errorMessage = '';
						this.closeOpenedModalIfAny();
						this.$dispatch('drawer.open');
					},
					onError: (error) => {
						this.$el.classList.remove('easystore-spinner');
						const response = error.response && JSON.parse(error.response);
						this.errorMessage = response?.data?.message ?? '';
						alert(this.errorMessage);
					},
				});
			},

			async openQuickAddModal(id) {
				this.$el.classList.add('easystore-spinner');
				const wrapper = document.querySelector('#easystore-modal');
				wrapper.classList.add('active');
				wrapper.setAttribute('aria-busy', 'true');
				const response = await fetch(
					`${Joomla.getOptions('easystore.base')}/index.php?option=com_easystore&task=product.quickAddModal&id=${id}`,
				);
				const data = await response.json();
				const content = wrapper.querySelector('.modal-content');
				content.innerHTML = data.data;
				wrapper.setAttribute('aria-busy', 'false');
				wrapper.setAttribute('aria-hidden', 'false');
				document.body.classList.add('easystore-modal-open');

				this.$el.classList.remove('easystore-spinner');
			},

			getVariant(combination, productId) {
				const refname = `product_variants_${productId}`;
				const variants = JSON.parse(this.$refs[refname].value);

				if (variants.length === 0 || !combination) {
					return null;
				}

				return variants.find((variant) => variant.combination.toLowerCase() === combination.toLowerCase()) ?? false;
			},

			generateCombination(options) {
				if (!Array.isArray(options)) {
					return;
				}

				options.sort();
				return options.join(';');
			},

			getCheckedRadios(productId) {
				const refname = `option_wrapper_${productId}`;
				if (!!this.$refs?.[refname]) {
					const checkedRadios = [...this.$refs[refname].querySelectorAll('input[type=radio]:checked')];

					if (checkedRadios.length === 0) {
						return [];
					}

					this.checkedOptions = checkedRadios.reduce((result, current) => {
						result[current.name] = current.value;
						return result;
					}, {});

					return checkedRadios.map((element) => element.value);
				}
			},

			async handleRadioChange(origin, productId) {
				const activeVariant = this.getVariant(this.generateCombination(this.getCheckedRadios(productId)), productId);
				const search = new URLSearchParams(window.location.search);

				if (origin === 'single') {
					search.set('variant', activeVariant.id);
					window.history.replaceState(null, '', `${window.location.pathname}?${search.toString()}`);
				}

				const section = origin === 'single' ? 'page-content' : 'quick-modal';
				await this.fetchProductContentByVariants(activeVariant.product_id, activeVariant.id, section);

				Joomla.applyImageLazyLoading();
			},

			async fetchProductContentByVariants(productId, variantId, section) {
				const url = `${Joomla.getOptions(
					'easystore.base',
				)}/index.php?option=com_easystore&task=product.contentByVariant&product_id=${productId}&variant_id=${variantId}&section_id=${section}`;
				const response = await fetch(url);
				const data = await response.json();

				const sectionElement = document.querySelector(data.data.section);
				sectionElement.innerHTML = data.data.content;
			},

			setInitialActiveSource() {
				const activeGalleryItem = document.querySelector('[easystore-gallery-item].active');

				if (activeGalleryItem) {
					this.activeSource = activeGalleryItem.dataset?.src ?? null;
					this.setCurrentSelectedIndex();
					this.watchSelectedImage(this.currentSelectedIndex);
				}
			},

			setCurrentSelectedIndex() {
				if (!this.activeSource || !this.imageSources.length) {
					this.currentSelectedIndex = 0;
					return;
				}

				this.currentSelectedIndex = this.imageSources.indexOf(this.activeSource);
			},

			get totalImages() {
				const wrapper = this.getRef('galleryThumbs');
				if (!wrapper) {
					return 0;
				}

				const images = [...wrapper.querySelectorAll('[easystore-zoom-thumb]')];
				return images.length;
			},

			get imageSources() {
				const wrapper = this.getRef('galleryThumbs');
				if (!wrapper) {
					return [];
				}

				return [...wrapper.querySelectorAll('[easystore-zoom-thumb]')]
					.map((image) => {
						return image.dataset?.src;
					})
					.filter((src) => !!src);
			},

			openGalleryPreview(type) {
				this.openPreview = true;
				this.activeMediaType = type;

				if (type === 'video') {
					const videoPreview = this.getRef('previewVideo');

					this.playVideo(videoPreview);
				}

				this.setInitialActiveSource();
			},

			isSelected(index) {
				return Number(index) === this.currentSelectedIndex;
			},

			selectPreviewImage() {
				this.currentSelectedIndex = Number(this.$el.dataset.index);
			},
			selectPreviousImage() {
				this.currentSelectedIndex = this.currentSelectedIndex - 1;
				if (this.currentSelectedIndex < 0) {
					this.currentSelectedIndex = this.totalImages - 1;
				}
			},
			selectNextImage() {
				this.currentSelectedIndex = this.currentSelectedIndex + 1;
				if (this.currentSelectedIndex >= this.totalImages) {
					this.currentSelectedIndex = 0;
				}
			},

			handlePlayPauseOnClick() {
				if (this.$el.paused) {
					this.$el.play();
				} else {
					this.$el.pause();
				}
			},

			playVideo(element) {
				if (!element) {
					return;
				}

				element.addEventListener('canplay', this.handleVideoPlay);
			},

			handleVideoPlay(event) {
				event.target.play();
			},
			pauseVideo(element) {
				if (!element) {
					return;
				}

				element.pause();
			},

			watchSelectedImage(index) {
				const thumb = this.getRef(`thumbItem${index}`);
				const src = thumb?.dataset?.src;
				const type = thumb?.dataset?.type;
				const previewVideo = this.getRef('previewVideo');
				const previewImage = this.getRef('previewImage');
				const previewElement = type === 'image' ? previewImage : previewVideo;

				this.playState = false;

				if (!src || !previewElement) {
					return;
				}

				this.activeMediaType = type;

				if (type === 'video') {
					previewVideo.closest('[easystore-preview-video]').style.display = 'block';
					previewImage.style.display = 'none';
					previewElement.style.cursor = 'pointer';
				} else {
					this.pauseVideo(previewVideo);
					previewVideo.closest('[easystore-preview-video]').style.display = 'none';
					previewImage.style.display = 'block';
					previewElement.style.cursor = 'zoom-in';
				}

				previewElement.src = src;
				previewElement.style.transform = `scale(1)`;
				this.zoomNextLevel = false;
			},
			watchNextLevelZooming(value) {
				const element = this.getRef('previewImage');

				if (!element) {
					return;
				}

				if (value) {
					element.style.transform = `scale(1.2)`;
					element.style.cursor = 'zoom-out';
				} else {
					element.style.transform = `scale(1)`;
					element.style.cursor = 'zoom-in';
				}
			},

			watchPreviewMode(value) {
				const gallery = this.getRef('zoomGallery');
				const backdrop = this.getRef('zoomBackdrop');
				const video = this.getRef('previewVideo');
				this.setCurrentSelectedIndex();

				if (!value) {
					video.removeEventListener('canplay', this.handleVideoPlay);
				}

				if (!gallery || !backdrop) {
					return;
				}

				if (value) {
					gallery.style.display = 'block';
					gallery.classList.add('gallery-opened');
					document.body.style.overflow = 'hidden';
					backdrop.style.display = 'block';
				} else {
					this.pauseVideo(video);
					gallery.style.display = 'none';
					document.body.style.overflow = 'auto';
					gallery.classList.remove('gallery-opened');
					backdrop.style.display = 'none';
				}

				if (!value && video) {
					video.pause();
				}
			},
		};
	});

	Alpine.data('easystoreProductList', () => {
		return {
			init() {
				this.pagination = Joomla.getOptions('easystore.pagination');
			},
			pagination: {},
		};
	});

	Alpine.data('easystorePaynow', () => ({
		selectedPaymentMethod: '',

		async generatePaymentSelectors() {
			const response = await fetch(
				`${Joomla.getOptions('easystore.base')}/index.php?option=com_easystore&task=order.render_payments`,
			);
			const data = await response.json();

			return data.data;
		},

		async openModal() {
			const wrapper = document.querySelector('#easystore-modal');
			const content = wrapper.querySelector('.modal-content');

			wrapper.classList.add('active');
			wrapper.setAttribute('aria-busy', 'true');

			const paymentSelectorContent = await this.generatePaymentSelectors();

			content.innerHTML = paymentSelectorContent;

			wrapper.setAttribute('aria-busy', 'false');
			wrapper.setAttribute('aria-hidden', 'false');

			document.body.classList.add('easystore-modal-open');
		},

		handlePaynow(orderDetails = '') {
			window.orderDetails = orderDetails;
			this.$el.classList.add('easystore-spinner');
			this.openModal();
			this.$el.classList.remove('easystore-spinner');
		},
	}));

	Alpine.data('easystorePaymentMethods', () => ({
		selectedPaymentMethod: '',
		handlePaymentChange(e) {
			this.selectedPaymentMethod = e.target.value;
		},

		async handlePaymentSubmit() {
			this.$el.classList.add('easystore-spinner');
			if (this.selectedPaymentMethod) {
				const orderDetails = window.orderDetails;
				orderDetails.payment_method = this.selectedPaymentMethod;
				const formData = new FormData();
				const acceptedFields = [
					'id',
					'creation_date',
					'customer_id',
					'customer_email',
					'shipping_address',
					'billing_address',
					'customer_note',
					'payment_status',
					'fulfilment',
					'order_status',
					'is_guest_order',
					'discount_type',
					'discount_value',
					'discount_reason',
					'shipping',
					'payment_method',
					'created',
					'created_by',
					'customerData',
				];
				const newObj = {};

				acceptedFields.forEach((acceptedField) => {
					if (orderDetails[acceptedField] !== undefined) {
						newObj[acceptedField] = orderDetails[acceptedField];
					}
				});

				newObj['customerData'] = { email: newObj['customerData'].email };

				formData.append('data', JSON.stringify(newObj));

				const url = `${Joomla.getOptions('easystore.base')}/index.php?option=com_easystore&task=checkout.orderRepay`;

				Joomla.request({
					url,
					method: 'POST',
					data: formData,
					onSuccess: (response) => {
						response = JSON.parse(response);
						if (response.success && !response?.data?.pluginError) {
							window.location.href = response.data.navigationUrl;
						} else {
							Joomla.renderMessages({ error: [response.data.pluginError] });
						}
					},
					onError: (xhr) => {
						Joomla.renderMessages(Joomla.ajaxErrorsMessages(xhr));
					},
					onComplete: () => {
						this.$el.classList.remove('easystore-spinner');
					},
				});
			}
		},
	}));
});

/** @TODO will be implemented the drag of zoomed image later. */
class DragImage {
	isDragging = false;
	startY = 0;
	startX = 0;
	translateY = 0;
	translateX = 0;
	element = null;

	clicked = false;
	isLeaving = false;

	constructor(element, threshold = 5) {
		this.element = element;
		this.threshold = threshold;
	}

	handleMouseDown(event) {
		this.isDragging = false;
		this.clicked = true;
		this.startY = event.clientY - this.element.offsetTop;
		this.startX = event.clientX - this.element.offsetLeft;
	}

	handleMouseMove(event) {
		const newX = event.clientX - this.startX;
		const newY = event.clientY - this.startY;

		if (!this.clicked) {
			return;
		}

		if (Math.abs(newX) > this.threshold || Math.abs(newY) > this.threshold) {
			this.isDragging = true;
		}

		if (this.isDragging) {
			this.element.style.transition = 'none';
			this.element.style.transform = `translate3d(${newX}px, ${newY}px, 0)`;
			this.translateX = newX;
			this.translateY = newY;
		}
	}

	handleMouseUp() {
		this.isDragging = false;
		this.clicked = false;
	}

	handleMouseEnter() {
		this.isLeaving = false;
	}
	handleMouseLeave() {
		this.isLeaving = true;

		if (this.isDragging) {
			this.isDragging = false;
			this.clicked = false;
			this.element.style.transition = 'transform 0.3s ese-out';
			this.element.style.transform = `translate3d(${this.translateX}px, ${this.translateY}px, 0)`;
		}
	}

	init() {
		this.element.addEventListener('mousedown', this.handleMouseDown.bind(this));
		this.element.addEventListener('mouseup', this.handleMouseUp.bind(this));
		this.element.addEventListener('mousemove', this.handleMouseMove.bind(this));
		this.element.addEventListener('mouseleave', this.handleMouseLeave.bind(this));
		this.element.addEventListener('mouseenter', this.handleMouseEnter.bind(this));
	}

	destroy() {
		this.element.removeEventListener('mousedown', this.handleMouseDown.bind(this));
		this.element.removeEventListener('mouseup', this.handleMouseUp.bind(this));
		this.element.removeEventListener('mousemove', this.handleMouseMove.bind(this));
		this.element.removeEventListener('mouseleave', this.handleMouseLeave.bind(this));
		this.element.removeEventListener('mouseenter', this.handleMouseEnter.bind(this));
	}
}
