<?php

/**
 * @package     EasyStore.Site
 * @subpackage  com_easystore
 *
 * @copyright   Copyright (C) 2023 - 2024 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Component\EasyStore\Site\Model;

use Joomla\CMS\Factory;
use Joomla\CMS\Router\Route;
use Joomla\Database\ParameterType;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\DatabaseInterface;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use JoomShaper\Component\EasyStore\Site\Traits\WishList;
use JoomShaper\Component\EasyStore\Site\Traits\ProductItem;
use JoomShaper\Component\EasyStore\Site\Helper\FilterHelper;
use JoomShaper\Component\EasyStore\Site\Helper\StringHelper;
use JoomShaper\Component\EasyStore\Site\Traits\Availability;
use JoomShaper\Component\EasyStore\Site\Traits\ProductMedia;
use JoomShaper\Component\EasyStore\Site\Traits\ProductOption;
use JoomShaper\Component\EasyStore\Site\Traits\ProductVariant;
use JoomShaper\Component\EasyStore\Administrator\Model\ProductModel;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreDatabaseOrm;
use JoomShaper\Component\EasyStore\Site\Helper\EasyStoreHelper as SiteEasyStoreHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreHelper as AdminEasyStoreHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

class ProductsModel extends ListModel
{
    use ProductMedia;
    use ProductOption;
    use ProductVariant;
    use Availability;
    use WishList;
    use ProductItem;

    /**
     * Model context string.
     *
     * @var    string
     * @since  1.0.0
     */
    public $_context = 'com_easystore.products';

    /**
     * Store the products query locally for using to the addons
     *
     * @var string
     */
    protected $productsQuery;

    public function __construct($config = [], MVCFactoryInterface $factory = null)
    {
        parent::__construct($config, $factory);
    }

    protected function populateState($ordering = 'ordering', $direction = 'ASC')
    {
        $app   = Factory::getApplication();
        $input = $app->getInput();

        // List state information
        $value = $input->get('limit', $app->get('list_limit', 0), 'uint');
        $this->setState('list.limit', $value);

        $value = $input->get('limitstart', 0, 'uint');
        $this->setState('list.start', $value);
    }

    /**
     * Method to build an SQL query to load the list data.
     *
     * @return  DatabaseQuery  An SQL query
     *
     * @since   1.0.0
     */
    protected function getListQuery()
    {
        $input = Factory::getApplication()->input;

        $catid                = null;
        $catIds               = $this->getCategoryIds($input->get('filter_categories', '', 'STRING'));
        $tagIds               = $this->getTagIds($input->get('filter_tags', '', 'STRING'));
        $filterVariants       = $this->getVariantNames($input->get('filter_variants', '', 'STRING'));
        $inventoryStatusValue = $this->getInventoryStatusValue($input->get('filter_inventory_status', '', 'STRING'));
        $min                  = $input->get('filter_min_price', 0, 'INT');
        $max                  = $input->get('filter_max_price', 100, 'INT');
        $search               = $input->get('filter_query', '', 'STRING');
        $sortBy               = 'best_selling';

        $attributes   = $this->getState('attr');
        $source       = $attributes['source'] ?? null;
        $easystorePks = $this->getState('easystore.pks', []);

        if (isset($attributes['start']) && isset($attributes['limit'])) {
            $this->setState('list.start', $attributes['start']);
            $this->setState('list.limit', $attributes['limit']);
        }

        if (!empty($attributes['catid'])) {
            $catAlias = $this->getAliasByCatId($attributes['catid']);
            $catid    = $this->getCategoryIds($catAlias);
        }

        $menuCatId = $input->get('catid', 0, 'INT');

        if (!empty($menuCatId)) {
            $catAlias = $this->getAliasByCatId($menuCatId);
            $catid    = $this->getCategoryIds($catAlias);
        }

        // Create the base query
        $query = $this->createBaseQuery($easystorePks);

        // Apply filters
        $query = $this->filterByCategory($query, $catid);
        $query = $this->filterByCategory($query, $catIds);
        $query = $this->filterByTags($query, $tagIds);
        $query = $this->filterByVariants($query, $filterVariants);
        $query = $this->filterByInventoryStatus($query, $inventoryStatusValue);
        $query = $this->filterByPriceRange($query, $min, $max);
        $query = $this->filterBySearchQuery($query, $search);

        if (!is_null($source)) {
            switch ($source) {
                case 'latest':
                    $sortBy = 'created-desc';
                    break;
                case 'oldest':
                    $sortBy = 'created-asc';
                    break;
                case 'on_sale':
                    $query = $this->filterByOnSale($query);
                    break;
                case 'best_selling':
                    $sortBy = 'best_selling';
                    break;
                case 'featured':
                    $query = $this->filterByFeatured($query);
                    break;
                case 'wishlist':
                    $query = $this->filterByWishlist($query);
                    break;
            }
        }

        $sortBy = $input->get('filter_sortby', $sortBy, 'STRING');
        $query  = $this->orderBy($query, $sortBy);


        if (!empty($attributes['pagination'])) {
            $params = (object) [
                'query' => clone $query,
                'limit' => $this->getState('list.limit', 20),
                'start' => $this->getState('list.start', 0),
            ];

            $this->loadPaginationStatusData($params);
        }

        return $query;
    }

    public function loadPaginationStatusData($data)
    {
        /** @var CMSApplication */
        $app      = Factory::getApplication();
        $document = $app->getDocument();
        $query    = $data->query;
        $limit    = $data->limit;
        $start    = $data->start;

        $query->clear('select')->clear('group')->clear('order')->select('count(distinct a.id) as total');
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $db->setQuery($query);
        $total = $db->loadResult() ?? 0;

        $pagination = (object) [
            'total'       => $total,
            'page'        => floor(($start + 1) / $limit) + 1,
            'total_pages' => ceil($total / $limit),
            'limit'       => $limit,
            'start'       => $start,
            'range_start' => $start + 1,
            'range_end'   => min($total, $start + $limit),
            'range'       => ($start + 1) . '-' . min($total, $start + $limit),
            'loaded'      => true,
        ];

        $document->addScriptOptions('easystore.pagination', $pagination);
    }

    /**
     * Get Category Alias form id
     *
     * @param int $catid
     * @return string|null
     */
    private function getAliasByCatId($catid)
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query->select('alias')
            ->from($db->quoteName('#__easystore_categories'))
            ->where($db->quoteName('id') . ' = ' . $catid);

        $db->setQuery($query);

        return $db->loadObject()->alias ?? null;
    }

    /**
     * Method to get an array of data items.
     *
     * @return  mixed  An array of data items on success, false on failure.
     *
     * @since   1.0.0
     */
    public function getItems()
    {
        $items           = new \stdClass();
        $items->products = parent::getItems();
        $orm             = new EasyStoreDatabaseOrm();
        $input           = Factory::getApplication()->input;

        $toRemoveProducts = [];

        $inventoryStatusValue = $this->getInventoryStatusValue($input->get('filter_inventory_status', '', 'STRING'));

        if (isset($items->products) && is_array($items->products)) {
            foreach ($items->products as &$item) {
                $item->link          = Route::_('index.php?option=com_easystore&view=product&id=' . $item->id . '&catid=' . $item->catid);
                $item->category_link = Route::_('index.php?option=com_easystore&view=products&catid=' . $item->catid);
                $item->media         = $this->getMedia($item->id);
                $item->options       = $this->getOptions($item->id);
                $item->variants      = $this->getVariants($item->id, $item);

                $variantCount = count($item->variants);
                $counter      = 0;

                if (!$inventoryStatusValue && $item->has_variants) {
                    foreach ($item->variants as $variant) {
                        if ($item->is_tracking_inventory) {
                            if ($variant->inventory_amount <= 0) {
                                $counter++;
                            }
                        } else {
                            if (!$variant->inventory_status) {
                                $counter++;
                            }
                        }
                    }

                    if ($counter != $variantCount) {
                        $toRemoveProducts[] = $item->id;
                    }
                }

                $item->discounted_price               = ($item->has_sale && $item->discount_value) ? AdminEasyStoreHelper::calculateDiscountedPrice($item->discount_type, $item->discount_value, $item->regular_price) : 0;
                $item->discounted_price_with_currency = AdminEasyStoreHelper::formatCurrency($item->discounted_price);
                $item->discounted_price_with_segments = AdminEasyStoreHelper::formatCurrency($item->discounted_price, true);
                $item->regular_price_with_currency    = AdminEasyStoreHelper::formatCurrency($item->regular_price);
                $item->regular_price_with_segments    = AdminEasyStoreHelper::formatCurrency($item->regular_price, true);
                $item->min_price_with_currency        = AdminEasyStoreHelper::formatCurrency($item->min_price);
                $item->min_price_with_segments        = AdminEasyStoreHelper::formatCurrency($item->min_price, true);

                $item->discounted_min_price = $item->has_sale && $item->discount_value > 0
                    ? AdminEasyStoreHelper::calculateDiscountedPrice($item->discount_type, $item->discount_value, $item->min_price)
                    : 0;
                $item->discounted_min_price_with_currency = AdminEasyStoreHelper::formatCurrency($item->discounted_min_price);
                $item->discounted_min_price_with_segments = AdminEasyStoreHelper::formatCurrency($item->discounted_min_price, true);

                $item->page       = 'productListPage';
                $item->reviewData = ProductModel::getReviewData($item->id);

                $tags = SiteEasyStoreHelper::getTags($item->id);

                if (isset($tags)) {
                    $tags = array_map(function ($tag) {
                        $tag['tag_link'] = Route::_('index.php?option=com_easystore&view=products&tags=' . $tag['tag_id']);
                        return $tag;
                    }, $tags);

                    $item->tags = $tags;
                }

                $app  = Factory::getApplication();
                $user = $app->getIdentity();

                if ($user->id) {
                    $item->inWishList = $this->isProductInWishlist($item->id, $user->id);
                }

                if ($item->has_variants) {
                    $variantPrice = $orm->setColumns([
                        $orm->aggregateQuoteName('MIN', 'price', 'min_price'),
                    ])
                        ->useRawColumns(true)
                        ->hasMany($item->id, '#__easystore_product_skus', 'product_id')
                        ->loadObject();

                    $item->discounted_price          = ($item->has_sale && $item->discount_value) ? AdminEasyStoreHelper::calculateDiscountedPrice($item->discount_type, $item->discount_value, (float) $variantPrice->min_price) : 0;
                    $item->currency_discounted_price = AdminEasyStoreHelper::formatCurrency($item->discounted_price, true);
                    $item->currency_regular_price    = AdminEasyStoreHelper::formatCurrency((float) $variantPrice->min_price, true);
                }

                $item->availability   = !empty($item->variants) ? $this->checkAvailability($item->variants, $item->options) : [];
                $item->active_variant = null;
                $item->thumbnail      = $this->getProductThumbnail($item);
                $item->prices         = $this->getPriceSegments($item, true);
                $item->stock          = $this->getProductStock($item);
            }

            unset($item);
        }

        if (!$inventoryStatusValue) {
            $newProducts = [];
            foreach ($items->products as $singleItem) {
                if (!in_array($singleItem->id, $toRemoveProducts)) {
                    $newProducts[] = $singleItem;
                }
            }

            $items->products = $newProducts;

            return $items;
        }

        return $items;
    }

    protected function filterBySearchQuery($query, $search)
    {
        if (empty($search)) {
            return $query;
        }

        $search = trim($search);

        $db           = $this->getDatabase();
        $search       = StringHelper::toRegexSafeString($search);
        $tagsSubQuery = $this->getProductsFromTags($search);

        $query->where($db->quoteName('a.title') . ' REGEXP ' . $db->quote($search) . ' OR ' . $db->quoteName('a.id') . ' IN (' . $tagsSubQuery . ')');

        return $query;
    }

    protected function getProductsFromTags($search)
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query->select('DISTINCT ptm.product_id')
            ->from($db->quoteName('#__easystore_tags', 't'))
            ->join('LEFT', $db->quoteName('#__easystore_product_tag_map', 'ptm') . ' ON (' . $db->quoteName('t.id') . ' = ' . $db->quoteName('ptm.tag_id') . ')')
            ->where($db->quoteName('t.title') . ' REGEXP ' . $db->quote($search));

        return $query;
    }

    /**
     * Sub Query to get the discounted price.
     *
     * @param   string   aParam  Param
     * @return  DatabaseQuery  An SQL query
     * @since   1.0.0
     */
    public function getSubQuery()
    {
        $db       = $this->getDatabase();
        $subQuery = $db->getQuery(true);

        $subQuery->select(
            [
                'CASE
                    WHEN (' . $db->quoteName('sub.has_sale') . ' = 1 AND ' . $db->quoteName('sub.discount_value') . ' > 0.00)
                        THEN
                            CASE
                                WHEN (' . $db->quoteName('sub.discount_type') . ' = "percent")
                                    THEN (' . $db->quoteName('sub.regular_price') . ' - (' . $db->quoteName('sub.regular_price') . ' * ' . $db->quoteName('sub.discount_value') . ') / 100)
                                ELSE (' . $db->quoteName('sub.regular_price') . ' - ' . $db->quoteName('sub.discount_value') . ')
                            END
                        ELSE sub.regular_price
                END AS filter_price',
            ]
        )
            ->from($db->quoteName('#__easystore_products', 'sub'))
            ->where($db->quoteName('a.id') . ' = ' . $db->quoteName('sub.id'));

        return $subQuery;
    }

    protected function createBaseQuery($pks = [])
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query
            ->select([
                'DISTINCT a.*',
                $db->quoteName('c.id', 'category_id'),
                $db->quoteName('c.title', 'category_title'),
                $db->quoteName('c.alias', 'category_alias'),
                '(CASE WHEN ' . $db->quoteName('a.has_variants') . ' = 1 THEN MIN(' . $db->quoteName('ps.price') . ') ELSE ' . $db->quoteName('a.regular_price') . ' END) as min_price',
            ])
            ->from($db->quoteName('#__easystore_products', 'a'))
            ->join('LEFT', $db->quoteName('#__easystore_categories', 'c'), $db->quoteName('c.id') . ' = ' . $db->quoteName('a.catid'))
            ->join('LEFT', $db->quoteName('#__easystore_product_skus', 'ps'), $db->quoteName('ps.product_id') . ' = ' . $db->quoteName('a.id'))
            ->where($db->quoteName('a.published') . ' = 1')
            ->where($db->quoteName('c.published') . ' = 1')
            ->group($db->quoteName('a.id'));

        if (!empty($pks)) {
            $query->where($db->quoteName('a.id') . ' IN (' . implode(',', $pks) . ')');
        }

        return $query;
    }

    protected function filterByCategory($query, $catIds)
    {
        if (empty($catIds)) {
            return $query;
        }

        if (!is_array($catIds)) {
            $catIds = [$catIds];
        }

        return $query->whereIn($this->getDatabase()->quoteName('a.catid'), $catIds);
    }

    protected function filterByTags($query, $tagIds)
    {
        if (!empty($tagIds)) {
            return $query->join('LEFT', $this->getDatabase()->quoteName('#__easystore_product_tag_map', 'pro_tag_map') . ' ON pro_tag_map.product_id = a.id')
                ->whereIn($this->getDatabase()->quoteName('pro_tag_map.tag_id'), $tagIds);
        }

        return $query;
    }

    protected function filterByVariants($query, $variantNames)
    {
        if (!empty($variantNames) && is_array($variantNames)) {
            return $query->join('INNER', $this->getDatabase()->quoteName('#__easystore_product_option_values', 'v') . ' ON ' . $this->getDatabase()->quoteName('a.id') . ' = ' . $this->getDatabase()->quoteName('v.product_id'))
                ->whereIn($this->getDatabase()->quoteName('v.name'), $variantNames, ParameterType::STRING);
        }

        return $query;
    }

    protected function filterByInventoryStatus($query, $inventoryStatusValue)
    {
        if ($inventoryStatusValue !== 'all') {
            $quantityOperator = '>';
            if (!$inventoryStatusValue) {
                $quantityOperator = '<=';
            }
            return $query->where(
                '((' . $this->getDatabase()->quoteName('a.has_variants') . ' = 0 AND ' .
                $this->getDatabase()->quoteName('a.is_tracking_inventory') . ' = 0 AND ' .
                $this->getDatabase()->quoteName('a.inventory_status') . ' = ' . (int) $inventoryStatusValue . ') OR (' .
                $this->getDatabase()->quoteName('a.has_variants') . ' = 0 AND ' .
                $this->getDatabase()->quoteName('a.is_tracking_inventory') . ' = 1 AND ' .
                $this->getDatabase()->quoteName('a.quantity') . $quantityOperator . ' 0 ) OR (' .
                $this->getDatabase()->quoteName('a.has_variants') . ' = 1 AND ' .
                $this->getDatabase()->quoteName('a.is_tracking_inventory') . ' = 0 AND ' .
                $this->getDatabase()->quoteName('ps.inventory_status') . ' = ' . (int) $inventoryStatusValue . ') OR (' .
                $this->getDatabase()->quoteName('a.has_variants') . ' = 1 AND ' .
                $this->getDatabase()->quoteName('a.is_tracking_inventory') . ' = 1 AND ' .
                $this->getDatabase()->quoteName('ps.inventory_amount') . $quantityOperator . ' 0))'
            );
        }

        return $query;
    }

    protected function filterByPriceRange($query, $min, $max)
    {
        if (!empty($min) && !empty(trim($max))) {
            return $query->where('(
                CASE
                    WHEN  ' . $this->getDatabase()->quoteName("a.has_variants") . ' = 0
                    THEN ' . $this->getDatabase()->quoteName("a.regular_price") . '
                    ELSE ' . $this->getDatabase()->quoteName("ps.price") . '
                END) BETWEEN ' . $min . ' AND ' . $max);
        }

        return $query;
    }

    protected function filterByWishlist($query)
    {
        $db   = $this->getDatabase();
        $user = Factory::getApplication()->getIdentity();

        if ($user->guest) {
            $query->where($db->quoteName('a.id') . ' = 0');
            return $query;
        }

        $subQuery = $db->getQuery(true);
        $subQuery->select($db->quoteName('wishlist.product_id'))
            ->from($db->quoteName('#__easystore_wishlist', 'wishlist'))
            ->where($db->quoteName('wishlist.user_id') . ' = ' . $db->quote($user->id));

        $query->where($db->quoteName('a.id') . ' IN (' . $subQuery . ')');

        return $query;
    }

    protected function filterByOnSale($query)
    {
        $query->where($this->getDatabase()->quoteName('a.has_sale') . ' = 1');

        return $query;
    }

    protected function filterByFeatured($query)
    {
        $query->where($this->getDatabase()->quoteName('a.featured') . ' = 1');

        return $query;
    }

    protected function orderByBestSellingProducts($query)
    {
        $db = $this->getDatabase();

        $query
            ->select('COUNT(DISTINCT opm.order_id) AS frequency')
            ->join('LEFT', $db->quoteName('#__easystore_order_product_map', 'opm') . ' ON (' . $db->quoteName('a.id') . ' = ' . $db->quoteName('opm.product_id') . ')')
            ->group($db->quoteName('a.id'))
            ->order($db->quoteName('frequency') . ' DESC')
            ->order($db->quoteName('created') . ' DESC');

        return $query;
    }

    protected function orderBy($query, $sortBy)
    {
        $db                  = $this->getDatabase();
        [$field, $direction] = FilterHelper::getOrder($sortBy);

        if ($field === 'price') {
            $column = $db->quoteName('min_price');
        } else {
            $column = $db->quoteName("a.{$field}");
        }

        switch ($field) {
            case 'best_selling':
                return $this->orderByBestSellingProducts($query);
            default:
                $query->order($column . ' ' . $direction);
                return $query;
        }

        return $query;
    }

    protected function getCategoryIds($filterCategories)
    {
        $filterCategories = $filterCategories ? explode(',', $filterCategories) : [$filterCategories];

        return (!empty($filterCategories) && is_array($filterCategories))
        ? $this->getCategoryIdsFromAliases($filterCategories)
        : [];
    }

    protected function getCategoryIdsFromAliases($aliases)
    {
        $db       = $this->getDatabase();
        $subQuery = $db->getQuery(true);

        $subQuery->select($db->quoteName('c.id'))
            ->from($db->quoteName('#__easystore_categories', 'c'))
            ->whereIn($db->quoteName('c.alias'), $aliases, ParameterType::STRING);

        $db->setQuery($subQuery);

        $catIds = $db->loadColumn();

        if (!empty($catIds)) {
            $allCategories = $this->getAllCategories();
            $parentIds     = [];

            foreach ($catIds as $catId) {
                $this->getCategoryIdList($allCategories, $catId, $parentIds);
            }

            return empty($parentIds) ? $catIds : array_merge($catIds, $parentIds);
        }

        return [];
    }

    public function getCategoryIdList($allCategories, $catid, &$ids)
    {
        $ids[] = $catid;

        foreach ($allCategories as $categories) {
            if ($categories->parent_id == $catid) {
                $this->getCategoryIdList($allCategories, $categories->id, $ids);
            }
        }
    }

    public function getAllCategories()
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select('cat.id, cat.parent_id')
            ->from($db->quoteName('#__easystore_categories', 'cat'))
            ->where($db->quoteName('cat.published') . ' = 1');

        $db->setQuery($query);

        return $db->loadObjectList();
    }

    protected function getTagIds($filterTags)
    {
        $filterTags = $filterTags ? explode(',', $filterTags) : [$filterTags];
        return (!empty($filterTags) && is_array($filterTags))
        ? $this->getTagIdsFromAliases($filterTags)
        : [];
    }

    protected function getTagIdsFromAliases($aliases)
    {
        $db       = $this->getDatabase();
        $subQuery = $db->getQuery(true);

        $subQuery->select($db->quoteName('t.id'))
            ->from($db->quoteName('#__easystore_tags', 't'))
            ->whereIn($db->quoteName('t.alias'), $aliases, ParameterType::STRING);

        $db->setQuery($subQuery);

        $tagIds = $db->loadColumn();

        return !empty($tagIds) ? $tagIds : [];
    }

    protected function getVariantNames($filterVariants)
    {
        $filterVariants = $filterVariants ? explode(',', $filterVariants) : $filterVariants;
        return (!empty($filterVariants) && is_array($filterVariants))
        ? $filterVariants
        : [];
    }

    protected function getInventoryStatusValue($inventoryStatus)
    {
        $inventoryStatusValue = ['out-of-stock' => 0, 'in-stock' => 1];

        return (!empty($inventoryStatus) && $inventoryStatus !== 'all')
        ? $inventoryStatusValue[$inventoryStatus]
        : 'all';
    }
}
