<?php

/**
 * @package     EasyStore.Site
 * @subpackage  com_easystore
 *
 * @copyright   Copyright (C) 2023 - 2024 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Component\EasyStore\Site\Model;

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\MVC\Model\ItemModel;
use Joomla\Database\DatabaseInterface;
use Joomla\CMS\MVC\View\GenericDataException;
use JoomShaper\Component\EasyStore\Site\Traits\ProductMedia;
use JoomShaper\Component\EasyStore\Site\Helper\EasyStoreHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\SettingsHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreDatabaseOrm;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreDatabaseMapHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreHelper as AdminHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class OrderModel extends ItemModel
{
    use ProductMedia;

    /**
     * Model context string.
     *
     * @var    string
     * @since  1.0.0
     */
    public $_context = 'com_easystore.order';

    /**
     * Method to auto-populate the model state.
     *
     * Note. Calling getState in this method will result in recursion.
     *
     * @since   1.6
     *
     * @return void
     */
    protected function populateState()
    {
        $app = Factory::getApplication();

        // Load state from the request.
        $pk = $app->getInput()->getInt('id');
        $this->setState('order.id', $pk);
    }

    /**
     * Method to get a single record.
     *
     * @param   int  $pk  The id of the primary key.
     *
     * @return  mixed
     *
     * @since   1.0.0
     */
    public function getItem($pk = null)
    {
        $loggedUser = $this->getCurrentUser();
        $user       = EasyStoreHelper::getCustomerByUserId($loggedUser->id);
        $pk         = (int) ($pk ?: $this->getState('order.id'));

        $isGuestOrder = 0;
        $guest        = new \stdClass();

        // Create a new query object.
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query->select(['o.*'])
            ->from($db->quoteName('#__easystore_orders', 'o'))
            ->where($db->quoteName('o.id') . ' = ' . $pk)
            ->where($db->quoteName('o.published') . ' = 1');

        $db->setQuery($query);

        try {
            $item = $db->loadObject();

            $isGuestOrder = $item->is_guest_order;

            if (empty($isGuestOrder) && is_null($item->customer_id) || empty($isGuestOrder) && is_null($user)) {
                throw new GenericDataException(Text::_("COM_EASYSTORE_ORDER_NOT_FOUND"), 404);
            }

            if (!empty($isGuestOrder)) {
                $guest->name                                 = '';
                $guest->email                                =  $item->customer_email;
                $guest->is_billing_and_shipping_address_same = false;
            }

            $query->clear();

            $subquery = $db->getQuery(true);
            $subquery->select($db->quoteName(['order_id', 'discount_type', 'discount_value', 'price', 'quantity']))
            ->select('SUM(' . $db->quoteName('price') . ' * ' . $db->quoteName('quantity') . ') AS sub_total')
            ->from($db->quoteName('#__easystore_order_product_map'))
            ->where($db->quoteName('order_id') . ' = ' . $pk)
            ->group([$db->quoteName('order_id'), $db->quoteName('discount_type'), $db->quoteName('discount_value'), $db->quoteName('price'), $db->quoteName('quantity')]);

            // Build the main query
            $query->select($db->quoteName(['order_id', 'discount_type', 'discount_value', 'price', 'quantity', 'sub_total']))
            ->select('
                CASE WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                    CASE 
                        WHEN ' . $db->quoteName('discount_type') . ' = ' . $db->quote('percent') . ' THEN 
                        ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                        ELSE ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                    END 
                ELSE
                    0.00
                END AS discounted_price,

                CASE
                    WHEN (
                        CASE
                            WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                                CASE
                                    WHEN ' . $db->quoteName('discount_type') . ' = "percent" THEN
                                       ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                                    ELSE ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                                END
                            ELSE
                                0.00
                        END
                    ) = 0.00 THEN 0.00
                    ELSE
                    ' . $db->quoteName('price') . ' - (
                            CASE
                                WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                                    CASE
                                        WHEN ' . $db->quoteName('discount_type') . ' = "percent" THEN
                                           ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                                        ELSE
                                           ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                                    END
                                ELSE
                                    0.00
                            END
                        )
                END  AS special_discounted_amount
                ')
            ->from('(' . $subquery . ') AS subquery');

            // Execute the query
            $db->setQuery($query);

            $orderedProducts = $db->loadObjectList();


            $item->sub_total                 = 0.00;
            $item->discounted_sub_total      = 0.00;
            $item->special_discounted_amount = 0.00;
            $item->extra_discount_amount     = 0.00;
            $item->shipping_cost             = 0.00;
            $item->sub_total_tax             = 0.00;
            $item->coupon_discount           = 0.00;
            $total_price                     = 0.00;


            if (!empty($orderedProducts)) {
                foreach ($orderedProducts as &$product) {
                    $item->sub_total += floatval($product->sub_total);

                    if (floatval($product->discounted_price) > 0) {
                        $item->discounted_sub_total += floatval($product->discounted_price);
                    }

                    if (!is_null(floatval($product->special_discounted_amount))) {
                        $item->special_discounted_amount += floatval($product->special_discounted_amount);
                    }

                    $item->sub_total -= $item->special_discounted_amount;
                }

                unset($product);
            }

            if (!is_null($item->coupon_code) && floatval($item->coupon_amount) > 0) {
                $item->coupon_discount = AdminHelper::calculateDiscountValue($item->coupon_type, $item->coupon_amount, $item->sub_total);
            }

            if (!is_null($item->discount_value) && floatval($item->discount_value) > 0) {
                $extra_discount = AdminHelper::calculateDiscountedPrice($item->discount_type, $item->discount_value, $item->sub_total);

                $item->extra_discount_amount = $item->sub_total - $extra_discount;
            }

            if (!empty($item->shipping) && is_string($item->shipping)) {
                // Convert the shipping string to an object
                $shippingData = json_decode($item->shipping);

                if (isset($shippingData->offerFreeShipping) && $shippingData->offerFreeShipping) {
                    // Check if there's an offer on a specific amount
                    $offerOnAmount = (float) ($shippingData->offerOnAmount ?? null);

                    if ($item->sub_total > $offerOnAmount) {
                        // Apply free shipping if the subtotal is above the offer amount
                        $shippingData->rate = 0;
                    }
                }

                if (isset($shippingData->rate)) {
                    // Format the shipping rate with currency
                    $shippingData->rate_with_currency = AdminHelper::formatCurrency($shippingData->rate);
                }

                if (isset($shippingData->rate) && empty($shippingData->rate)) {
                    // Set the shipping name to "Free Shipping" if the rate is empty
                    $shippingData->name = Text::_('COM_EASYSTORE_FREE_SHIPPING');
                }

                // Update the shipping cost with the calculated rate or default to 0
                $item->shipping_cost = $shippingData->rate ?? 0;
            }


            if (!empty($item->sale_tax)) {
                $item->sub_total_tax = $item->sub_total + floatval($item->sale_tax);
            }

            $total_price = $item->sub_total + $item->shipping_cost + $item->sale_tax - $item->coupon_discount - $item->extra_discount_amount;

            $item->products                                = $this->getProducts($item->id);
            $item->sub_total_with_currency                 = AdminHelper::formatCurrency($item->sub_total);
            $item->sub_total_tax_with_currency             = AdminHelper::formatCurrency($item->sub_total_tax);
            $item->special_discounted_amount_with_currency = AdminHelper::formatCurrency($item->special_discounted_amount);
            $item->extra_discount_amount_with_currency     = AdminHelper::formatCurrency($item->extra_discount_amount);
            $item->total_price_with_currency               = AdminHelper::formatCurrency($total_price);
            $item->shipping_cost_with_currency             = AdminHelper::formatCurrency($item->shipping_cost);
            $item->coupon_discount_with_currency           = AdminHelper::formatCurrency($item->coupon_discount);
            $item->sale_tax_with_currency                  = AdminHelper::formatCurrency($item->sale_tax);

            $orm = new EasyStoreDatabaseOrm();

            foreach ($item->products as $product) {
                $product->canReview         = empty($isGuestOrder) ? EasyStoreHelper::canReview($user->id, $product->id) : false;
                $product->hasGivenReview    = (empty($isGuestOrder) && is_null($user->id)) ? EasyStoreHelper::hasGivenReview($user->id, $product->id) : true;
                $product->imageSrc          = $product->media->thumbnail->src ?? '';

                if (floatval($product->discount_value) > 0) {
                    $product->price = AdminHelper::calculateDiscountedPrice($product->discount_type, $product->discount_value, $product->price);
                }

                $totalPrice = floatval($product->quantity * $product->price);

                $product->priceWithQuantity = AdminHelper::formatCurrency($totalPrice);
                $product->actualPrice       = AdminHelper::formatCurrency($product->price);
                $combinationName            = $orm->hasOne($product->variant_id, '#__easystore_product_skus', 'id')->loadObject();
                $product->options           = !empty($combinationName) ? $combinationName->combination_name : '';
            }
        } catch (\Throwable $e) {
            throw new \Exception($e->getMessage());
        }

        if (empty($isGuestOrder) && !empty($user->shipping_address)) {
            $user->shipping_address = $this->generateAddress($user->shipping_address);
        } else {
            $guest->shipping_address = $this->generateAddress($item->shipping_address);
        }

        if (empty($isGuestOrder) && !empty($user->billing_address)) {
            $user->billing_address = $this->generateAddress($user->billing_address);
        } else {
            $guest->billing_address = $this->generateAddress($item->billing_address);
        }

        $item->customerData = empty($isGuestOrder) ? $user : $guest;

        $settings    = SettingsHelper::getSettings();

        $item->storeAddress             = new \stdClass();
        $item->storeAddress->address1   = $settings->get('general.addressLineOne', '');
        $item->storeAddress->address2   = $settings->get('general.addressLineTwo', '');
        $item->storeAddress->country    = $settings->get('general.country', '');
        $item->storeAddress->state      = $settings->get('general.state', '');
        $item->storeAddress->city       = $settings->get('general.city', '');
        $item->storeAddress->zip        = $settings->get('general.postcode', '');
        $item->history                  = $this->getOrderHistory($pk);

        return $item;
    }

    private function getOrderHistory($orderId)
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query->select($db->quoteName(['o.id', 'o.activity_type', 'o.created']))
            ->from($db->quoteName('#__easystore_order_activities', 'o'))
            ->where($db->quoteName('o.order_id') . ' = ' . $orderId)
            ->where($db->quoteName('o.activity_type') . ' != ' . $db->quote('comment'))
            ->order($db->quoteName('o.created') . 'DESC');

        $db->setQuery($query);

        $result   = $db->loadObjectList();
        $newArray = [];

        foreach ($result as $data) {
            $newArray[] = (object) [
                'activity_type' => EasyStoreDatabaseMapHelper::getOrderActivities($data->activity_type),
                'created'       => HTMLHelper::_('date', $data->created, Text::_("COM_EASYSTORE_DATE_FORMAT_1")),
            ];
        }

        return $newArray;
    }

    public function generateAddress($address)
    {
        $addressObject          = json_decode($address);
        $countryState           = EasyStoreHelper::getCountryStateFromJson($addressObject->country, $addressObject->state);
        $addressObject->country = $countryState->country;
        $addressObject->state   = $countryState->state;

        return $addressObject;
    }


    /**
     * Get product details of an order
     *
     * @param  int  $id     Order ID
     * @return CMSObject|bool  Object on success, false on failure.
     * @since  1.0.0
     */
    public function getProducts($id)
    {
        // Create a new query object.
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query->select($db->quoteName(['p.id', 'p.title']))
            ->from($db->quoteName('#__easystore_products', 'p'))
            ->where($db->quoteName('p.published') . ' = 1');

        $query->select($db->quoteName(['ord_pro_map.quantity', 'ord_pro_map.discount_type', 'ord_pro_map.discount_value', 'ord_pro_map.price', 'ord_pro_map.variant_id']))
            ->join('LEFT', $db->quoteName('#__easystore_order_product_map', 'ord_pro_map'), $db->quoteName('ord_pro_map.product_id') . ' = ' . $db->quoteName('p.id'))
            ->where($db->quoteName('ord_pro_map.order_id') . ' = ' . $id);


        $db->setQuery($query);

        try {
            $result = $db->loadObjectList();

            foreach ($result as $product) {
                $media          = $this->getMedia($product->id);
                $product->media = $media;
            }

            return $result;
        } catch (\Throwable $e) {
            throw new \Exception($e->getMessage());
        }
    }

    public function getItemForReorder($orderId, $country = null, $state = null)
    {
        $orm                 = new EasyStoreDatabaseOrm();
        $settings            = SettingsHelper::getSettings();
        $weightUnit          = $settings->get('products.standardUnits.weight', 'kg');
        $isTaxEnabled        = $settings->get('checkout.enable_tax', true);
        $isCouponCodeEnabled = $settings->get('checkout.enable_coupon_code', true);

        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $conditions = [
            $db->quoteName('id') . ' = ' . $orderId,
        ];

        $query->select('*')->from($db->quoteName('#__easystore_orders'))
            ->where($conditions);
        $db->setQuery($query);

        try {
            $order = $db->loadObject();
        } catch (\Exception $error) {
            $order = null;
        }

        if (empty($order)) {
            return null;
        }

        $columns = [
            'oi.quantity',
            'product.id as product_id',
            'product.title',
            'product.alias',
            'product.regular_price',
            'product.has_sale',
            'product.quantity as available_quantity',
            'product.is_tracking_inventory',
            'product.inventory_status as product_inventory_status',
            'product.quantity as product_inventory_amount',
            'product.discount_type',
            'product.discount_value',
            'product.enable_out_of_stock_sell',
            'product.has_variants',
            'product.weight',
            'sku.id as sku_id',
            'sku.combination_value',
            'sku.price as price',
            'sku.weight as sku_weight',
            'sku.image_id',
            'sku.inventory_status as sku_inventory_status',
            'sku.inventory_amount as sku_inventory_amount',
        ];

        $order->items = $orm->hasMany($order->id, ['#__easystore_order_product_map', 'oi'], 'order_id')
            ->updateQuery(function ($query) use ($db, $columns) {
                $query->select($columns)->join(
                    'LEFT',
                    $db->quoteName('#__easystore_products', 'product') . ' ON ' . $db->quoteName('product.id') . ' = ' . $db->quoteName('oi.product_id')
                )->join(
                    'LEFT',
                    $db->quoteName('#__easystore_product_skus', 'sku') . ' ON ' . $db->quoteName('sku.id') . ' = ' . $db->quoteName('oi.variant_id')
                );
            })->loadObjectList();

        $order->sub_total    = 0;
        $order->total_weight = 0;

        if (!empty($order->items)) {
            foreach ($order->items as &$item) {
                if (!empty($item->image_id)) {
                    $item->image = $orm->setColumns(['id', 'src'])
                        ->hasOne($item->image_id, '#__easystore_media', 'id')
                        ->loadObject();
                } else {
                    $item->image = $orm->setColumns(['id', 'src'])
                        ->hasOne($item->product_id, '#__easystore_media', 'product_id')
                        ->updateQuery(function ($query) use ($db) {
                            $query->where($db->quoteName('is_featured') . ' = 1');
                        })->loadObject();
                }

                if (!empty($item->image) && !empty($item->image->src)) {
                    $item->image->src = Path::clean(Uri::root(true) . '/' . $item->image->src);
                }

                if ($item->has_variants) {
                    $item->inventory_amount = $item->sku_inventory_amount ?? 0;
                    $item->inventory_status = $item->sku_inventory_status ?? 0;
                    $item->price            = $item->price ?? $item->regular_price;

                    $weight                 = $item->sku_weight ?? 0;
                    $item->weight           = (float) $weight;
                    $item->weight_with_unit = $item->sku_weight . $weightUnit;
                } else {
                    $item->price            = $item->regular_price ?? 0;
                    $item->inventory_amount = $item->available_quantity ?? 0;
                    $item->inventory_status = $item->product_inventory_status ?? 0;

                    $weight                 = $item->weight ?? 0;
                    $item->weight           = (float) $weight;
                    $item->weight_with_unit = $item->weight . $weightUnit;
                }

                $order->total_weight += $item->weight * (int) $item->quantity;

                $item->discount = (object) [
                    'type'   => $item->discount_type ?? 'percent',
                    'amount' => $item->discount_value ?? 0,
                ];

                unset(
                    $item->discount_type,
                    $item->discount_value,
                    $item->regular_price,
                    $item->sku_inventory_amount,
                    $item->product_inventory_amount,
                    $item->sku_inventory_status,
                    $item->product_inventory_status,
                    $item->available_quantity,
                    $item->image_id
                );

                $item->discounted_price = 0;
                $item->item_price       = 0;
                $item->item_price       = $item->price;

                if ($item->has_sale && $item->discount->amount > 0) {
                    $item->discounted_price = AdminHelper::calculateDiscountedPrice($item->discount->type, $item->discount->amount, $item->price);
                    $item->item_price       = $item->discounted_price;
                }

                $item->total_price                    = floatval($item->discounted_price) > 0 ? $item->discounted_price * $item->quantity : $item->price * $item->quantity;
                $item->total_price_with_currency      = AdminHelper::formatCurrency($item->total_price);
                $item->price_with_currency            = AdminHelper::formatCurrency($item->price);
                $item->discounted_price_with_currency = AdminHelper::formatCurrency($item->discounted_price);

                $cart          = new CartModel();
                $item->options = $cart->detectProductOptionFromCombination(
                    $cart->getProductOptionsById($item->product_id),
                    $item->combination_value
                );

                $order->sub_total += $item->total_price ?? 0;
            }

            unset($item);
        }

        $settingsModel   = new SettingsModel();
        $taxRate         = $settingsModel->getTaxRate($country, $state);
        $shippingMethods = $settingsModel->getShipping($country, $state);

        if (!empty($order->shipping) && is_string($order->shipping)) {
            $order->shipping = json_decode($order->shipping);
        }

        if (empty($order->shipping) && !empty($shippingMethods)) {
            $order->shipping = reset($shippingMethods);
        }

        if (isset($order->shipping->offerFreeShipping) && $order->shipping->offerFreeShipping) {
            $offerOnAmount = $order->shipping->offerOnAmount ?? null;

            if (!is_null($offerOnAmount)) {
                $offerOnAmount = (float) $offerOnAmount;

                if ($order->sub_total > $offerOnAmount) {
                    $order->shipping->rate = 0;
                }
            }
        }

        if (isset($order->shipping->rate)) {
            $order->shipping->rate_with_currency = AdminHelper::formatCurrency($order->shipping->rate);
        }

        if (isset($order->shipping->rate) && empty($order->shipping->rate)) {
            $order->shipping->name = Text::_('COM_EASYSTORE_FREE_SHIPPING');
        }

        $order->shipping_price  = $order->shipping->rate ?? 0;
        $order->coupon_discount = 0;
        $order->coupon_amount   = (float) $order->coupon_amount;

        if (!empty($order->coupon_code) && $order->coupon_amount > 0) {
            $order->coupon_discount = AdminHelper::calculateDiscountValue($order->coupon_type, $order->coupon_amount, $order->sub_total);
        }

        $order->tax_rate                      = !empty($taxRate) ? $taxRate->rate : 0;
        $order->coupon_discount_with_currency = AdminHelper::formatCurrency($order->coupon_discount);
        $order->sub_total_with_currency       = AdminHelper::formatCurrency($order->sub_total);
        $order->taxable_amount                = number_format($order->sale_tax, 2);

        $order->total = $order->sub_total + $order->shipping_price;

        if ($isTaxEnabled) {
            $order->total += $order->taxable_amount;
        }

        if ($isCouponCodeEnabled) {
            $order->total -= $order->coupon_discount;
        }

        $order->taxable_amount_with_currency  = AdminHelper::formatCurrency($order->taxable_amount);
        $order->total_with_currency           = AdminHelper::formatCurrency($order->total);
        $order->total_weight_with_unit        = $order->total_weight . $weightUnit;

        return $order;
    }
}
