<?php

/**
 * @package     EasyStore.Site
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Site\Controller;

use Throwable;
use Joomla\Input\Json;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Event\AbstractEvent;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\MVC\Controller\BaseController;
use JoomShaper\Component\EasyStore\Site\Traits\Api;
use JoomShaper\Component\EasyStore\Site\Traits\Token;
use JoomShaper\Component\EasyStore\Site\Helper\EasyStoreHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Default Controller of EasyStore component
 *
 * @since  1.0.0
 */
class PaymentController extends BaseController
{
    use Api;
    use Token;

    /**
     * After a successful payment, the following functionalities will be triggered.
     *
     * @since 1.0.0
     */
    public function onPaymentSuccess()
    {
        (new CheckoutController())->removeCartData();

        /** @var CMSApplication */
        $app         = Factory::getApplication();
        $input       = $app->getInput();
        $paymentType = $input->getString('type');
        $orderID     = $input->getString('order_id');
        $uri         = Uri::getInstance(Route::_('index.php?option=com_easystore&view=payment&layout=success', false));

        if (!empty($paymentType)) {
            $uri->setVar('type', $paymentType);
        }

        if (!empty($orderID)) {
            $uri->setVar('order_id', $orderID);
        }

        $app->redirect($uri->toString());
    }

    /**
     * If the payment is canceled, the following functionalities will be triggered.
     * @since 1.0.0
     * @return void
     */
    public function onPaymentCancel()
    {
        /** @var CMSApplication */
        $app     = Factory::getApplication();
        $input   = $app->getInput();
        $orderID = $input->get('order_id', '', 'STRING');

        $app->redirect(Route::_('index.php?option=com_easystore&view=payment&layout=cancel&order_id=' . $orderID, false));
    }

    /**
     * This function manages payment notifications. It identifies the payment type, imports the relevant
     * plugin, and triggers the 'onPaymentNotify' event. If any errors occur, a user-friendly message is
     * displayed.
     *
     * @since 1.0.0
     */
    public function onPaymentNotify()
    {
        $paymentType = $this->input->getString('type');
        PluginHelper::importPlugin('easystore', $paymentType);

        $rawPayLoad      = (new Json())->getRaw();
        $postData        = Factory::getApplication()->input->post->getArray();
        $serverVariables = Factory::getApplication()->input->server->getArray();
        $getData         = Factory::getApplication()->input->get->getArray();
        $order           = new CheckoutController();

        $event = AbstractEvent::create(
            'onPaymentNotify',
            [
                'subject' => (object) [
                    'raw_payload'      => $rawPayLoad,
                    'get_data'         => $getData,
                    'post_data'        => $postData,
                    'server_variables' => $serverVariables,
                    'order'            => $order,
                ],
            ]
        );

        try {
            Factory::getApplication()->getDispatcher()->dispatch($event->getName(), $event);
        } catch (\Throwable $th) {
            Factory::getApplication()->enqueueMessage($th->getMessage());
        }
    }

    /**
     * Redirects to the payment gateway based on the provided payment data.
     * Decodes the 'data' parameter, imports the necessary easystore plugin,
     * retrieves cart data for payment, and triggers the 'onPayment' event.
     * Handles errors during event dispatching by displaying an error message
     * and redirecting back to the checkout page.
     *
     * @since 1.0.3
     */

    public function navigateToPaymentGateway()
    {
        $data        = Factory::getApplication()->input->get('data', '', 'STRING');
        $data        = json_decode(base64_decode($data));
        $data->token = $this->getToken();

        PluginHelper::importPlugin('easystore', $data->payment_method);

        if (isset($data->order_type) && $data->order_type === 'reorder') {
            $dataForPayment = EasyStoreHelper::getOrderDataForPayment($data);
        } else {
            $dataForPayment = EasyStoreHelper::getCartDataForPayment($data);
        }

        $event = AbstractEvent::create(
            'onPayment',
            [
                'subject' => (object) $dataForPayment,
            ]
        );

        try {
            Factory::getApplication()->getDispatcher()->dispatch($event->getName(), $event);
        } catch (Throwable $error) {
            Factory::getApplication()->enqueueMessage($error->getMessage(), 'error');
            Factory::getApplication()->redirect($dataForPayment->back_to_checkout_page);
        }
    }
}
