<?php

/**
 * @package     EasyStore.Site
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Site\Controller;

use Exception;
use Throwable;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\Input\Input;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Event\AbstractEvent;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Database\DatabaseInterface;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use JoomShaper\Component\EasyStore\Site\Traits\Api;
use JoomShaper\Component\EasyStore\Site\Traits\Token;
use JoomShaper\Component\EasyStore\Site\Model\CartModel;
use JoomShaper\Component\EasyStore\Site\Traits\Checkout;
use JoomShaper\Component\EasyStore\Site\Helper\ArrayHelper;
use JoomShaper\Component\EasyStore\Site\Model\CheckoutModel;
use JoomShaper\Component\EasyStore\Site\Model\SettingsModel;
use JoomShaper\Component\EasyStore\Site\Helper\EasyStoreHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\SettingsHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreDatabaseOrm;
use JoomShaper\Component\EasyStore\Administrator\Model\OrderModel as AdminOrderModel;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Checkout Controller of EasyStore component
 *
 * @since  1.0.0
 */
class CheckoutController extends BaseController
{
    use Api;
    use Checkout;
    use Token;

    public function __construct($config = [], MVCFactoryInterface $factory = null, ?CMSApplication $app = null, ?Input $input = null)
    {
        parent::__construct($config, $factory, $app, $input);
    }

    /**
     * Place an Order for guest and shop customer.
     *
     * @return void
     * @since 1.0.0
     */
    public function placeOrder()
    {
        $app                  = Factory::getApplication();
        $user                 = $app->getIdentity();
        $input                = $app->input;
        $token                = $this->getToken();
        $settings             = SettingsHelper::getSettings();
        $isPhoneFieldEnabled  = $settings->get('checkout.phone_number');
        $allowGuestCheckout   = $settings->get('checkout.allow_guest_checkout', false);
        $emailTemplates       = $settings->get('email_templates', '');
        $customer             = EasyStoreHelper::getCustomerByUserId($user->id);
        $isGuestCheckout      = $user->guest && $allowGuestCheckout;
        $saveGuestInformation = (bool) $input->get('save_address', false);
        $customerEmail        = $isGuestCheckout ? $input->get('email', '', 'email') : $user->email;
        $customerNote         = $input->get('customer_note', '', 'text');
        $pluginError          = null;
        $navigationUrl        = null;
        $isEmailEnabled       = false;

        // Check is email templates enable
        if (!empty($emailTemplates) && is_array($emailTemplates->order->templates)) {
            foreach ($emailTemplates->order->templates as $template) {
                if (!empty($template->type) && $template->type === "order_confirmation") {
                    $isEmailEnabled = $template->is_enabled;
                }
            }
        }

        // Load related models
        $cartModel     = $this->getModel('Cart', 'Site');
        $checkoutModel = $this->getModel('Checkout', 'Site');

        $cartItems = $cartModel->getItem();

        if (empty($cartItems->items)) {
            $this->sendResponse(['message' => Text::_("COM_EASYSTORE_CART_EMPTY"), 'code' => 500]);
        }

        $cartData = (object) [
            'token'          => $token,
            'payment_method' => $input->get('payment_method', '', 'RAW'),
            'cart_items'     => $cartItems,
        ];

        $shippingAddress = json_decode($input->get('shipping_address', '', 'RAW') ?? "[]");

        if (!empty($shippingAddress)) {
            $this->validateField($shippingAddress->name, "COM_EASYSTORE_CHECKOUT_ERROR_CUSTOMER_NAME");
            $this->validateField($shippingAddress->address_1, "COM_EASYSTORE_CHECKOUT_ERROR_ADDRESS_LINE_1");
            $this->validateField($shippingAddress->country, "COM_EASYSTORE_CHECKOUT_ERROR_COUNTRY");
            $this->validateField($shippingAddress->city, "COM_EASYSTORE_CHECKOUT_ERROR_CITY");
            $this->validateField($shippingAddress->zip_code, "COM_EASYSTORE_CHECKOUT_ERROR_ZIP_CODE");

            if ($isPhoneFieldEnabled === "required") {
                $this->validateField($shippingAddress->phone, "COM_EASYSTORE_CHECKOUT_ERROR_CUSTOMER_NUMBER");
            }
        }

        if (!$input->get('is_billing_and_shipping_address_same', 0, 'INT')) {
            $billingAddress = json_decode($input->get('billing_address', '', 'RAW') ?? "[]");

            if (!empty($billingAddress)) {
                $this->validateField($billingAddress->name, "COM_EASYSTORE_CHECKOUT_ERROR_CUSTOMER_NAME");
                $this->validateField($billingAddress->address_1, "COM_EASYSTORE_CHECKOUT_ERROR_ADDRESS_LINE_1");
                $this->validateField($billingAddress->country, "COM_EASYSTORE_CHECKOUT_ERROR_COUNTRY");
                $this->validateField($billingAddress->city, "COM_EASYSTORE_CHECKOUT_ERROR_CITY");
                $this->validateField($billingAddress->zip_code, "COM_EASYSTORE_CHECKOUT_ERROR_ZIP_CODE");

                if ($isPhoneFieldEnabled === "required") {
                    $this->validateField($billingAddress->phone, "COM_EASYSTORE_CHECKOUT_ERROR_CUSTOMER_NUMBER");
                }
            }
        }

        $information = (object) [
            'user_id'                              => $user->id ?? 0,
            'shipping_address'                     => $input->get('shipping_address', '', 'RAW'),
            'is_billing_and_shipping_address_same' => $input->get('is_billing_and_shipping_address_same', 0, 'INT'),
            'billing_address'                      => $input->get('billing_address', '', 'RAW'),
        ];

        $customerPhoneNumber = isset($customer->phone) ? $customer->phone : "";
        $phoneNumber         = empty($customerPhoneNumber) ? $shippingAddress->phone : $customerPhoneNumber;

        switch ($isPhoneFieldEnabled) {
            case 'optional':
                $information->phone = $phoneNumber;
                break;
            case 'do-not-include':
                $information->phone = $customer->phone ?? "";
                break;
            case 'required':
                $information->phone = $phoneNumber;
                break;
        }

        $shippingData = (object) [
            'token'           => $token,
            'shipping_method' => $input->get('shipping_method', '', 'RAW'),
        ];

        if ($information->is_billing_and_shipping_address_same) {
            $information->billing_address = $information->shipping_address;
        }

        try {
            $orm = new EasyStoreDatabaseOrm();
            $db  = Factory::getContainer()->get(DatabaseInterface::class);

            $db->transactionStart();

            if ($isGuestCheckout) {
                if ($saveGuestInformation) {
                    $guestData = (object) [
                        'email'            => $customerEmail,
                        'shipping_address' => $information->shipping_address,
                    ];

                    $checkoutModel->saveGuestCustomerInformation($guestData);
                }
            } else {
                if (empty($customer->id) && !empty($phoneNumber)) {
                    $information->phone = $phoneNumber;
                }

                $checkoutModel->saveCustomerInformation($information);
            }

            $this->saveShipping($shippingData);

            $orm->update('#__easystore_cart', $cartData, 'token');

            if (empty($cartData->payment_method)) {
                $pluginError = Text::sprintf('COM_EASYSTORE_NO_PAYMENT_METHOD_FOUND', ucfirst($cartData->payment_method), 'error');

                $this->sendResponse(['success' => true, 'navigationUrl' => $navigationUrl, 'pluginError' => $pluginError]);
            }

            if ($cartData->payment_method !== 'manual_payment') {
                if (PluginHelper::isEnabled('easystore', $cartData->payment_method)) {
                    // Load payment plugin to check the plugin required fields
                    PluginHelper::importPlugin('easystore', $cartData->payment_method);

                    // Verify all the required plugin fields are filled in or not.
                    $requiredFieldsStatus = $this->checkRequiredFields();

                    if (!$requiredFieldsStatus) {
                        $pluginError   = Text::_('COM_EASYSTORE_PAYMENT_PLUGIN_ERROR');
                        $this->sendResponse(['success' => true, 'navigationUrl' => $navigationUrl, 'pluginError' => $pluginError]);
                    }
                } else {
                    $pluginError = Text::sprintf('COM_EASYSTORE_CART_PLUGIN_IS_DISABLE', ucfirst($cartData->payment_method), 'error');
                    $this->sendResponse(['success' => true, 'navigationUrl' => $navigationUrl, 'pluginError' => $pluginError]);
                }
            }

            $orderData = (object) [
                'order_status'      => 'active',
                'email'             => $customerEmail,
                'shipping_address'  => $information->shipping_address,
                'billing_address'   => $information->billing_address,
                'is_guest_checkout' => $isGuestCheckout,
                'customer_note'     => $customerNote,
            ];

            $order = $this->createOrder($orderData);

            // Data for payment plugins
            $paymentData = (object) [
                'user_id'          => $user->id,
                'guest'            => $isGuestCheckout,
                'customer_email'   => $customerEmail,
                'shipping_address' => $shippingAddress,
                'order_id'         => $order->id,
                'payment_method'   => $cartData->payment_method,
                'billing_address'  => $information->billing_address,
            ];

            if ($cartData->payment_method !== 'manual_payment') {
                if (PluginHelper::isEnabled('easystore', $cartData->payment_method)) {
                    $navigationUrl = Route::_(Uri::base() . 'index.php?option=com_easystore&task=payment.navigateToPaymentGateway&data=' . base64_encode(json_encode($paymentData)), false);
                } else {
                    $pluginError = Text::sprintf('COM_EASYSTORE_CART_PLUGIN_IS_DISABLE', ucfirst($cartData->payment_method), 'error');
                }
            } else {
                $navigationUrl = Route::_(Uri::base() . 'index.php?option=com_easystore&task=payment.onPaymentSuccess&type=' . $cartData->payment_method, false);
            }

            $db->transactionCommit();

            $storeAddress = SettingsHelper::getAddress();

            $variables = [
                'order_id'         => $order->id,
                'order_summary'    => LayoutHelper::render('emails.order.summary', EasyStoreHelper::getOrderSummary($order->id)),
                'order_date'       => $order->creation_date,
                'payment_status'   => EasyStoreHelper::getPaymentStatusString($order->payment_status),
                'payment_method'   => $order->payment_method === "manual_payment" ? Text::_("COM_EASYSTORE_PAYMENT_COD_TITLE") : ucwords($order->payment_method),
                'shipping_address' => LayoutHelper::render('emails.address', EasyStoreHelper::convertAddressCountryStateData($order->shipping_address)),
                'order_link'       => Route::_(Uri::root() . 'index.php?option=com_easystore&view=order&id=' . $order->id),
                'customer_name'    => $user->name ?? Text::_('COM_EASYSTORE_ORDER_BY_CUSTOMER'),
                'store_name'       => $settings->get('general.storeName', ''),
                'store_email'      => $settings->get('general.storeEmail', ''),
                'store_phone'      => $settings->get('general.storePhone', ''),
                'store_address'    => LayoutHelper::render('emails.address', $storeAddress),
            ];

            if (PluginHelper::isEnabled('system', 'easystoremail')) {
                PluginHelper::importPlugin('system', 'easystoremail');

                if ($isEmailEnabled && $order->payment_method === 'manual_payment') {
                    $event = AbstractEvent::create('onOrderPlaced', [
                        'subject' => (object) [
                            'variables'      => $variables,
                            'type'           => 'order_confirmation',
                            'customer_email' => $customerEmail,
                        ],
                    ]);
                    Factory::getApplication()->getDispatcher()->dispatch($event->getName(), $event);
                }
            }

            $this->sendResponse(['success' => true, 'navigationUrl' => $navigationUrl, 'pluginError' => $pluginError]);
        } catch (Exception $error) {
            $db->transactionRollback();
            $this->sendResponse(['message' => $error->getMessage(), 'code' => 500]);
        }
    }

    /**
     * Function to Repay failed or canceled order
     *
     * @return void
     * @since 1.0.7
     */
    public function orderRepay()
    {
        $app                = Factory::getApplication();
        $user               = $app->getIdentity();
        $input              = $app->input;
        $data               = json_decode($input->get('data', '[]', 'RAW'));
        $settings           = SettingsHelper::getSettings();
        $allowGuestCheckout = $settings->get('checkout.allow_guest_checkout', false);
        $emailTemplates     = $settings->get('email_templates', '');
        $isGuestCheckout    = $user->guest && $allowGuestCheckout;
        $customerEmail      = $data->customerData->email;
        $pluginError        = null;
        $navigationUrl      = null;
        $isEmailEnabled     = false;

        // Check is email templates enable
        if (!empty($emailTemplates) && is_array($emailTemplates->order->templates)) {
            foreach ($emailTemplates->order->templates as $template) {
                if (!empty($template->type) && $template->type === "order_confirmation") {
                    $isEmailEnabled = $template->is_enabled;
                }
            }
        }

        $order      = (object) [
            'id'               => $data->id,
            'creation_date'    => $data->creation_date,
            'customer_id'      => $data->customer_id,
            'customer_email'   => $data->customer_email,
            'shipping_address' => $data->shipping_address,
            'billing_address'  => $data->billing_address,
            'customer_note'    => $data->customer_note ?? "",
            'payment_status'   => $data->payment_status,
            'fulfilment'       => $data->fulfilment,
            'order_status'     => $data->order_status,
            'is_guest_order'   => $data->is_guest_order ? 1 : 0,
            'discount_type'    => $data->discount_type,
            'discount_value'   => $data->discount_value,
            'discount_reason'  => $data->discount_reason,
            'shipping'         => $data->shipping,
            'payment_method'   => $data->payment_method,
            'created'          => $data->created,
            'created_by'       => $data->created_by,
        ];

        // Data for payment plugins
        $paymentData = (object) [
            'user_id'          => $user->id,
            'guest'            => $isGuestCheckout,
            'customer_email'   => $customerEmail,
            'shipping_address' => json_decode($data->shipping_address),
            'order_id'         => $order->id,
            'payment_method'   => $data->payment_method,
            'order_type'       => 'reorder',
            'billing_address'  => json_decode($data->billing_address),
        ];

        if (empty($data->payment_method)) {
            $pluginError = Text::sprintf('COM_EASYSTORE_NO_PAYMENT_METHOD_FOUND', ucfirst($data->payment_method), 'error');

            $this->sendResponse(['success' => true, 'navigationUrl' => $navigationUrl, 'pluginError' => $pluginError]);
        }

        if ($data->payment_method !== 'manual_payment') {
            if (PluginHelper::isEnabled('easystore', $data->payment_method)) {
                $navigationUrl = Route::_(Uri::base() . 'index.php?option=com_easystore&task=payment.navigateToPaymentGateway&data=' . base64_encode(json_encode($paymentData)), false);

                // Load payment plugin to check the plugin required fields
                PluginHelper::importPlugin('easystore', $data->payment_method);

                // Verify all the required plugin fields are filled in or not.
                $requiredFieldsStatus = $this->checkRequiredFields();

                if (!$requiredFieldsStatus) {
                    $pluginError   = Text::_('COM_EASYSTORE_PAYMENT_PLUGIN_ERROR');
                    $navigationUrl = null;
                }
            } else {
                $pluginError = Text::sprintf('COM_EASYSTORE_CART_PLUGIN_IS_DISABLE', ucfirst($data->payment_method), 'error');
            }
        } else {
            $navigationUrl = Route::_(Uri::base() . 'index.php?option=com_easystore&task=payment.onPaymentSuccess&type=' . $data->payment_method, false);

            $this->updateOrder($order);
        }

        $storeAddress = SettingsHelper::getAddress();

        $variables = [
            'order_id'         => $order->id,
            'order_summary'    => LayoutHelper::render('emails.order.summary', EasyStoreHelper::getOrderSummary($order->id)),
            'order_date'       => $order->creation_date,
            'payment_status'   => EasyStoreHelper::getPaymentStatusString($order->payment_status),
            'payment_method'   => $order->payment_method === "manual_payment" ? Text::_("COM_EASYSTORE_PAYMENT_COD_TITLE") : ucwords($order->payment_method),
            'shipping_address' => LayoutHelper::render('emails.address', EasyStoreHelper::convertAddressCountryStateData($order->shipping_address)),
            'order_link'       => Route::_(Uri::root() . 'index.php?option=com_easystore&view=order&id=' . $order->id),
            'customer_name'    => $user->name ?? Text::_('COM_EASYSTORE_ORDER_BY_CUSTOMER'),
            'store_name'       => $settings->get('general.storeName', ''),
            'store_email'      => $settings->get('general.storeEmail', ''),
            'store_phone'      => $settings->get('general.storePhone', ''),
            'store_address'    => LayoutHelper::render('emails.address', $storeAddress),
        ];

        if (PluginHelper::isEnabled('system', 'easystoremail')) {
            PluginHelper::importPlugin('system', 'easystoremail');

            if ($isEmailEnabled && $order->payment_method === 'manual_payment') {
                $event = AbstractEvent::create('onOrderPlaced', [
                    'subject' => (object) [
                        'variables'      => $variables,
                        'type'           => 'order_confirmation',
                        'customer_email' => $customerEmail,
                    ],
                ]);
                Factory::getApplication()->getDispatcher()->dispatch($event->getName(), $event);
            }
        }

        $this->sendResponse(['success' => true, 'navigationUrl' => $navigationUrl, 'pluginError' => $pluginError]);
    }

    /**
     * Get default country and state for a customer
     *
     * @return array
     * @since 1.0.0
     */
    protected function getDefaultCountryState()
    {
        $customer = EasyStoreHelper::getCustomerByUserId(Factory::getApplication()->getIdentity()->id);

        $defaultCountry = null;
        $defaultState   = null;

        if (!empty($customer->shipping_address)) {
            $shippingAddress = is_string($customer->shipping_address) ? json_decode($customer->shipping_address) : null;

            if ($shippingAddress) {
                $defaultCountry = $shippingAddress->country ?? null;
                $defaultState   = $shippingAddress->state ?? null;
            }
        }

        return [$defaultCountry, $defaultState];
    }

    /**
     * Get all the cart items for showing cart summary in checkout view.
     *
     * @return void
     * @since 1.0.0
     */
    public function getCartData()
    {
        [$defaultCountry, $defaultState] = $this->getDefaultCountryState();
        $cartModel                       = new CartModel();
        $shippingId                      = Factory::getApplication()->input->get('shipping_id', null, 'STRING');
        $country                         = Factory::getApplication()->input->get('country', $defaultCountry, 'STRING');
        $state                           = Factory::getApplication()->input->get('state', $defaultState, 'STRING');
        $cart                            = $cartModel->getItem($shippingId, $country, $state);

        $this->sendResponse($cart);
    }

    /**
     * Get shipping and billing address with customer information.
     *
     * @return void
     * @since 1.0.0
     */
    public function getInformation()
    {
        $model       = new CheckoutModel();
        $information = $model->getInformation();

        $this->sendResponse($information);
    }

    /**
     * Get store default checkout settings values.
     *
     * @return void
     * @since 1.0.0
     */
    public function getCheckoutSettings()
    {
        $settings        = SettingsHelper::getSettings();
        $checkoutSetting = $settings->get('checkout', null) ?? (object) [];

        $this->sendResponse($checkoutSetting);
    }

    /**
     * Get store default shipping address
     *
     * @return void
     * @since 1.0.0
     */
    public function getShipping()
    {
        [$defaultCountry, $defaultState] = $this->getDefaultCountryState();

        $input    = Factory::getApplication()->input;
        $country  = $input->get('country', $defaultCountry);
        $state    = $input->get('state', $defaultState);
        $subtotal = $input->get('subtotal', null);

        $settingModel = new SettingsModel();

        $this->sendResponse($settingModel->getShipping($country, $state, $subtotal) ?? []);
    }

    public function getPaymentMethods()
    {
        $this->sendResponse($this->getActivePayments());
    }

    public function saveShipping($data)
    {
        $orm = new EasyStoreDatabaseOrm();

        try {
            $orm->update('#__easystore_cart', $data, 'token');

            return true;
        } catch (Throwable $error) {
            throw $error;
        }
    }

    public function searchGuestUser()
    {
        $input = Factory::getApplication()->input;
        $email = $input->get('email', '', 'email');

        $model    = new CheckoutModel();
        $shipping = $model->getGuestShippingAddress($email);

        $this->sendResponse($shipping);
    }

    public function onOrderPlacementCompletion()
    {
        $this->deductQuantityFromInventory();
        $this->removeCartData();
    }

    public function deductQuantityFromInventory()
    {
        $cart = (new CartModel())->getItem();

        if (empty($cart)) {
            return;
        }

        if (empty($cart->items)) {
            return;
        }

        foreach ($cart->items as $item) {
            $data = (object) [
                'product_id' => $item->product_id,
                'sku_id'     => $item->sku_id,
                'quantity'   => $item->quantity,
            ];

            $this->deductFromProductOrSkuTable($data);
        }
    }

    public function deductFromProductOrSkuTable($data)
    {
        $db    = Factory::getDbo();
        $query = $db->getQuery(true);
        $query->select('is_tracking_inventory')->from($db->quoteName('#__easystore_products'))
            ->where($db->quoteName('id') . ' = ' . (int) $data->product_id);

        $db->setQuery($query);

        try {
            $isTrackingInventory = (bool) ($db->loadResult() ?? 0);
        } catch (Throwable $error) {
            throw $error;
        }

        if (!$isTrackingInventory) {
            return;
        }

        $tableName  = !empty($data->sku_id) ? '#__easystore_product_skus' : '#__easystore_products';
        $columnName = !empty($data->sku_id) ? 'inventory_amount' : 'quantity';

        $fields = [
            $db->quoteName($columnName) . ' = ' . $db->quoteName($columnName) . ' - ' . $data->quantity,
        ];

        $conditions = [
            $db->quoteName('id') . ' = ' . $db->quote(!empty($data->sku_id) ? $data->sku_id : $data->product_id),
        ];

        $query = $db->getQuery(true)->update($tableName)
            ->set($fields)
            ->where($conditions);

        $db->setQuery($query);

        try {
            $db->execute();
        } catch (Throwable $error) {
            throw $error;
        }
    }

    public function removeCartData()
    {
        $token = $this->getToken();

        $db    = Factory::getDbo();
        $query = $db->getQuery(true);

        $query->delete($db->quoteName('#__easystore_cart'))
            ->where($db->quoteName('token') . ' = ' . $db->quote($token));

        $db->setQuery($query);

        try {
            $db->execute();
            $this->removeToken();
        } catch (Throwable $error) {
            throw $error;
        }
    }

    public function createOrder($data)
    {
        [$defaultCountry, $defaultState] = $this->getDefaultCountryState();
        $shippingAddress                 = Factory::getApplication()->input->get('shipping_address', '', 'RAW');

        if (!empty($shippingAddress) && is_string($shippingAddress)) {
            $shippingAddress = json_decode($shippingAddress);
        }

        $country = $shippingAddress->country ?? $defaultCountry;
        $state   = $shippingAddress->state ?? $defaultState;

        $user            = Factory::getApplication()->getIdentity();
        $currentDateTime = Factory::getDate('now')->toSql();
        $customer        = EasyStoreHelper::getCustomerByUserId($user->id);
        $cartModel       = new CartModel();
        $cartItem        = $cartModel->getItem(null, $country, $state);

        $settings            = SettingsHelper::getSettings();
        $isTaxEnabled        = $settings->get('checkout.enable_tax', true);
        $isCouponCodeEnabled = $settings->get('checkout.enable_coupon_code', true);

        $data = (object) [
            'id'               => null,
            'creation_date'    => $currentDateTime,
            'customer_id'      => $customer->id ?? null,
            'customer_email'   => $data->email,
            'shipping_address' => $data->shipping_address,
            'billing_address'  => $data->billing_address,
            'customer_note'    => $data->customer_note,
            'payment_status'   => 'unpaid',
            'fulfilment'       => 'unfulfilled',
            'order_status'     => $data->order_status,
            'is_guest_order'   => $data->is_guest_checkout ? 1 : 0,
            'discount_type'    => 'percent',
            'discount_value'   => 0,
            'discount_reason'  => '',
            'shipping'         => !is_string($cartItem->shipping_method) ? json_encode($cartItem->shipping_method) : '',
            'payment_method'   => $cartItem->payment_method,
            'created'          => $currentDateTime,
            'created_by'       => $user->id,
        ];

        if ($isTaxEnabled) {
            $data->sale_tax = $cartItem->taxable_amount ?? 0.00;
        }

        if ($isCouponCodeEnabled) {
            $data->coupon_code   = $cartItem->coupon_code ?? null;
            $data->coupon_type   = $cartItem->coupon_type ?? null;
            $data->coupon_amount = $cartItem->coupon_amount ?? 0.00;
        }

        $orm = new EasyStoreDatabaseOrm();

        try {
            $orm->create('#__easystore_orders', $data, 'id');
            $this->addOrderItems($cartItem->items, $data->id);

            $adminOrderModel = new AdminOrderModel();
            $adminOrderModel->addOrderActivity($data->id, 'order_created');

            return $data;
        } catch (Throwable $error) {
            throw $error;
        }
    }

    public function updateOrder($data)
    {
        $orm = new EasyStoreDatabaseOrm();

        try {
            $orm->update('#__easystore_orders', $data, 'id');

            $settings     = SettingsHelper::getSettings();
            $storeName    = $settings->get('general.storeName', '');
            $storeEmail   = $settings->get('general.storeEmail', '');
            $storePhone   = $settings->get('general.storePhone', '');
            $storeAddress = SettingsHelper::getAddress();

            $order        = $orm->get('#__easystore_orders', 'id', $data->id)->loadObject();

            $variables       = [
                'order_id'         => $order->id,
                'order_date'       => $order->creation_date,
                'payment_status'   => EasyStoreHelper::getPaymentStatusString($order->payment_status),
                'payment_method'   => $order->payment_method,
                'shipping_address' => LayoutHelper::render('emails.address', EasyStoreHelper::convertAddressCountryStateData($order->shipping_address)),
                'order_link'       => Route::_(Uri::root() . 'index.php?option=com_easystore&view=order&id=' . $order->id),
                'store_name'       => $storeName,
                'store_email'      => $storeEmail,
                'store_phone'      => $storePhone,
                'store_address'    => LayoutHelper::render('emails.address', $storeAddress),
            ];

            $type = 'payment_success';

            if (in_array($data->payment_status, ['failed','canceled'])) {
                $type = 'payment_error';

                $event = AbstractEvent::create('onFailedPayment', [
                    'subject' => (object) [
                        'variables'      => $variables,
                        'customer_email' => $order->customer_email,
                        'type'           => $type,
                    ],

                ]);

                Factory::getApplication()->getDispatcher()->dispatch($event->getName(), $event);
            } else {
                $event = AbstractEvent::create('onSuccessfulPayment', [
                    'subject' => (object) [
                        'variables'      => $variables,
                        'customer_email' => $order->customer_email,
                        'type'           => $type,
                    ],
                ]);

                Factory::getApplication()->getDispatcher()->dispatch($event->getName(), $event);

                $event = AbstractEvent::create('onSuccessfulPayment', [
                    'subject' => (object) [
                        'variables'      => $variables,
                        'customer_email' => $order->customer_email,
                        'type'           => 'order_confirmation',
                    ],
                ]);

                Factory::getApplication()->getDispatcher()->dispatch($event->getName(), $event);
            }

            return true;
        } catch (Throwable $error) {
            throw $error;
        }

        return false;
    }

    public function addOrderItems($items, $orderId)
    {
        if (empty($items)) {
            return;
        }

        $orm = new EasyStoreDatabaseOrm();

        foreach ($items as $item) {
            $data = (object) [
                'order_id'        => $orderId,
                'product_id'      => $item->product_id,
                'variant_id'      => $item->sku_id ?? null,
                'discount_type'   => $item->discount->type ?? 'percent',
                'discount_value'  => $item->discount->value ?? 0,
                'discount_reason' => '',
                'quantity'        => $item->quantity,
                'price'           => $item->item_price,
            ];

            try {
                $orm->create('#__easystore_order_product_map', $data);
            } catch (Throwable $error) {
                throw $error;
            }
        }

        return true;
    }

    public function applyCoupon()
    {
        $input  = Factory::getApplication()->input;
        $code   = $input->get('code', '', 'STRING');
        $cartId = $input->get('cart_id', 0, 'INT');

        if (!$code || !$cartId) {
            $this->sendResponse(['message' => 'Missing required information.'], 400);
        }

        $model = new CheckoutModel();

        try {
            $coupon = $model->getCouponByCode($code);
        } catch (Exception $error) {
            $this->sendResponse(['message' => $error->getMessage()], 500);
        }

        if (is_null($coupon)) {
            $this->sendResponse(['message' => Text::_('COM_EASYSTORE_CART_COUPON_INVALID')], 404);
        }

        if (!EasyStoreHelper::isCouponCodeValid($coupon->start_date, $coupon->end_date)) {
            $this->sendResponse(['message' => Text::_('COM_EASYSTORE_CART_COUPON_CODE_EXPIRED')], 400);
        }

        $couponData = (object) [
            'code'   => $coupon->code,
            'type'   => $coupon->discount_type,
            'amount' => $coupon->discount_value,
        ];

        try {
            $model->applyCouponCode($cartId, $couponData);
        } catch (Exception $error) {
            $this->sendResponse(['message' => $error->getMessage()], 500);
        }

        $this->sendResponse(['success' => true]);
    }

    public function removeCode()
    {
        $input  = Factory::getApplication()->input;
        $cartId = $input->get('cart_id', 0, 'INT');

        if (!$cartId) {
            $this->sendResponse(['message' => 'Missing required information.'], 400);
        }

        $model = new CheckoutModel();

        try {
            $model->removeCouponCode($cartId);
        } catch (Exception $error) {
            $this->sendResponse(['message' => $error->getMessage()], 500);
        }

        $this->sendResponse(['success' => true]);
    }

    private function loadCountriesData()
    {
        $jsonPath  = JPATH_ROOT . '/media/com_easystore/data/countries.json';
        $countries = [];

        if (file_exists($jsonPath)) {
            $countries = file_get_contents($jsonPath);

            if (!empty($countries) && is_string($countries)) {
                $countries = json_decode($countries);
            }
        }

        return $countries;
    }

    public function getCountries()
    {
        $countriesData = $this->loadCountriesData();

        $settingsModel     = new SettingsModel();
        $countryWithStates = $settingsModel->getCountriesWithStates();

        $codes = array_keys($countryWithStates);

        $countries = ArrayHelper::findByArray(function ($haystack, $item) {
            return ArrayHelper::find(function ($value) use ($item) {
                return $value->numeric_code == $item;
            }, $haystack);
        }, $countriesData, $codes);

        $countries = ArrayHelper::toOptions(function ($country) {
            return (object) [
                'label' => $country->name,
                'value' => $country->numeric_code,
            ];
        }, $countries);

        $this->sendResponse($countries);
    }

    protected function isAllStatesAllowed($countryCode)
    {
        $settings = SettingsHelper::getSettings();
        $shipping = $settings->get('shipping', []);

        $isAllowed = false;

        foreach ($shipping as $item) {
            if (!empty($item->regions)) {
                foreach ($item->regions as $region) {
                    if ($region->country == $countryCode && empty($region->states)) {
                        $isAllowed = true;
                        break;
                    }
                }
            }

            if ($isAllowed) {
                break;
            }
        }

        return $isAllowed;
    }

    public function getStates()
    {
        $input       = Factory::getApplication()->input;
        $countryCode = $input->get('country_code', '');

        $settingsModel     = new SettingsModel();
        $countryWithStates = $settingsModel->getCountriesWithStates();

        if (empty($countryCode) || empty($countryWithStates)) {
            $this->sendResponse([]);
        }

        $countriesData = $this->loadCountriesData();
        $country       = ArrayHelper::find(function ($item) use ($countryCode) {
            return $item->numeric_code == $countryCode;
        }, $countriesData);

        if (is_null($country)) {
            $this->sendResponse([]);
        }

        $states = $country->states ?? [];

        if (empty($states)) {
            $this->sendResponse([]);
        }

        if ($this->isAllStatesAllowed($countryCode)) {
            $options = ArrayHelper::toOptions(function ($item) {
                return (object) [
                    'label' => $item->name,
                    'value' => $item->id,
                ];
            }, $states);

            $this->sendResponse($options);
        }

        $stateCodes = $countryWithStates[$countryCode] ?? [];

        if (empty($stateCodes)) {
            $this->sendResponse([]);
        }

        $states = ArrayHelper::findByArray(function ($haystack, $item) {
            return ArrayHelper::find(function ($value) use ($item) {
                return $value->id == $item;
            }, $haystack);
        }, $states, $stateCodes);

        $options = ArrayHelper::toOptions(function ($item) {
            return (object) [
                'label' => $item->name,
                'value' => $item->id,
            ];
        }, $states);

        $this->sendResponse($options);
    }

    /**
     * Check if all the required fields for the plugin are filled.
     *
     * @return bool  The result of the check, indicating whether the required fields are filled.
     * @since 1.0.3
     */

    public function checkRequiredFields()
    {
        $event = AbstractEvent::create(
            'onBeforePayment',
            [
                'subject' => new \stdClass(),
            ]
        );

        try {
            $eventResult = Factory::getApplication()->getDispatcher()->dispatch($event->getName(), $event);

            return $eventResult->getArgument('result');
        } catch (Throwable $error) {
            Factory::getApplication()->enqueueMessage($error->getMessage(), 'error');
            return false;
        }
    }

    /**
     * Check valid field
     *
     * @param string $value Field to check the value.
     *
     * @return void
     *
     * @since 1.0.7
     */
    public function validateField(string $value, string $message)
    {

        $input = trim($value);

        if (empty($input)) {
            $this->sendResponse(
                [
                    'message' => Text::_($message),
                    'code'    => 500,
                ]
            );
        }
    }
}
