<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Administrator\Table;

use Joomla\CMS\Factory;
use Joomla\CMS\Table\Table;
use Joomla\Database\DatabaseDriver;
use JoomShaper\Component\EasyStore\Administrator\Model\OrderModel;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Orders table
 *
 * @since  1.0.0
 */
class OrderTable extends Table
{
    /**
     * Indicates that columns fully support the NULL value in the database
     *
     * @var    bool
     * @since  1.0.0
     */
    protected $_supportNullValue = true;

    /**
     * Constructor
     *
     * @param   DatabaseDriver  $db  A database connector object
     */
    public function __construct(DatabaseDriver $db)
    {
        parent::__construct('#__easystore_orders', 'id', $db);
    }

    /**
     * Overloaded check method to ensure data integrity.
     *
     * @return  bool  True on success.
     *
     * @since   1.0.0
     * @throws  \UnexpectedValueException
     */
    public function check()
    {
        try {
            parent::check();
        } catch (\Exception $e) {
            $this->setError($e->getMessage());

            return false;
        }

        if (!(int) $this->checked_out_time) {
            $this->checked_out_time = null;
        }

        return true;
    }

    /**
     * Overridden \JTable::store to set modified data and user id.
     *
     * @param   bool  $updateNulls  True to update fields even if they are null.
     *
     * @return  bool  True on success.
     *
     * @since   1.0.0
     */
    public function store($updateNulls = true)
    {
        $date = Factory::getDate();
        $user = Factory::getApplication()->getIdentity();

        if ($this->id) {
            // Existing item
            $this->modified_user_id = $user->get('id');
            $this->modified_time    = $date->toSql();
        } else {
            // New order. A order created and created_by field can be set by the user,
            // so we don't touch either of these if they are set.
            if (!(int) $this->created_time) {
                $this->created_time = $date->toSql();
            }

            if (empty($this->created_user_id)) {
                $this->created_user_id = $user->get('id');
            }

            if (!(int) $this->modified_time) {
                $this->modified_time = $this->created_time;
            }

            if (empty($this->modified_user_id)) {
                $this->modified_user_id = $this->created_user_id;
            }
        }

        if (!empty($this->shipping_type)) {
            $chunk                = explode(':', $this->shipping_type);
            $this->shipping_value = (float) $chunk[0];
            $this->shipping_type  = $chunk[1];
        }

        // Will work later
        $this->order_shipping_address_id = 0;
        $this->order_billing_address_id  = 0;

        return parent::store($updateNulls);
    }

    /**
     * Function to generate Unique ID
     *
     * @return string
     */
    private function generateUniqueId()
    {
        $isUnique = false;

        while (!$isUnique) {
            $uniqueId = uniqid();
            $uniqueId = '#' . strtoupper($uniqueId);
            $isUnique = OrderModel::isTrackingIdUnique($uniqueId);
        }

        return $uniqueId;
    }
}
