<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Administrator\Plugin;

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Router\Route;
use Joomla\Registry\Registry;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Application\CMSApplication;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


/**
 * Constants Class
 *
 * This class encapsulates constants and configuration settings related to the EasyStore payment plugin.
 * @since 1.0.5
 */
class Constants
{
    /**
     * The payment plugin name
     *
     * @var string
     */
    protected $name;

    /**
     * Plugin params
     *
     * @var Registry
     */
    protected $params;

    /** @var CMSApplication */
    protected $app;

    /**
     * Constructor method for the payment plugin constants
     *
     * @param string $name the plugin name
     * @since 1.0.5
     */
    public function __construct(string $name)
    {
        $this->name   = $name;
        $plugin       = PluginHelper::getPlugin('easystore', $this->name);
        $this->params = new Registry($plugin->params);
        $this->app    = Factory::getApplication();
    }

    /**
     *  Returns a URL to which the user will be redirected upon completing the payment procedure on the payment portal.
     *
     * @param  int|null $orderID  -- Order ID(optional).
     * @return string
     * @since  1.0.5
     */
    public function getSuccessUrl($orderID = null)
    {
        $uri = new Uri(Route::_(Uri::root() . 'index.php?option=com_easystore&task=payment.onPaymentSuccess'));
        $uri->setVar('type', $this->name);

        if (!is_null($orderID)) {
            $uri->setVar('order_id', $orderID);
        }

        return $uri->toString();
    }

    /**
     * Returns the cancellation url if the user cancel the payment on the payment portal.
     *
     * @return string
     * @since  1.0.5
     */
    public function getCancelUrl($orderID)
    {
        return Route::_(Uri::root() . 'index.php?option=com_easystore&task=payment.onPaymentCancel&order_id=' . $orderID, false);
    }

    /**
     * Returns the webhook URL which will be used to receive notifications from payment portal.
     *
     * @return string
     * @since  1.0.5
     */
    public function getWebHookUrl()
    {
        return Route::_(Uri::root() . 'index.php?option=com_easystore&task=payment.onPaymentNotify&type=' . $this->name, false);
    }

    /**
     * Retrieve the payment environment if a `payment environment` field is configured in the settings.
     *
     * @return string
     * @since  1.0.6
     */

    public function getPaymentEnvironment()
    {
        return $this->params->get('payment_environment', 'test');
    }

    /**
     * Returns a test Key if a `test key` field is configured in the settings.
     *
     * @return string
     * @since  1.0.6
     */

    public function getTestKey()
    {
        return $this->params->get('test_key', '');
    }

    /**
     * Returns a Live Key if a `live key` field is configured in the settings.
     *
     * @return string
     * @since  1.0.6
     */

    public function getLiveKey()
    {
        return $this->params->get('live_key', '');
    }

    /**
     * Returns the secret key or an alternative based on the payment environment and the values set in the `test_key` and `live_key` fields.
     *
     * @return string
     * @since  1.0.6
     */

    public function getSecretKey()
    {
        return $this->getPaymentEnvironment() === 'test' ? self::getTestKey() : self::getLiveKey();
    }
}
