<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Administrator\Model;

use Joomla\Database\DatabaseQuery;
use Joomla\Database\ParameterType;
use Joomla\CMS\MVC\Model\ListModel;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * This models supports retrieving a list of orders.
 *
 * @since  1.0.0
 */
class OrdersModel extends ListModel
{
    /**
     * Constructor.
     *
     *
     * @param   array   $config   An optional associative array of configuration settings.
     *
     * @since   1.0.0
     */
    public function __construct($config = [])
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = [
                'id',
                'a.id',
                'customer_name',
                'payment_status',
                'a.payment_status',
                'fulfilment',
                'a.fulfilment',
                'published',
                'a.published',
                'a.access',
                'access_title',
                'a.created',
                'created',
                'order_status',
                'a.order_status',
            ];
        }

        parent::__construct($config);
    }

    /**
     * Method to auto-populate the model state.
     *
     * Note. Calling getState in this method will result in recursion.
     *
     * @param   string  $ordering   An optional ordering field.
     * @param   string  $direction  An optional direction (asc|desc).
     *
     * @return  void
     *
     * @since   1.0.0
     */
    protected function populateState($ordering = 'a.id', $direction = 'desc')
    {
        $search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
        $this->setState('filter.search', $search);

        $published = $this->getUserStateFromRequest($this->context . '.filter.published', 'filter_published', '');
        $this->setState('filter.published', $published);

        $paymentStatus = $this->getUserStateFromRequest($this->context . '.filter.payment_status', 'filter_payment_status', '');
        $this->setState('filter.payment_status', $paymentStatus);

        $fulfilment = $this->getUserStateFromRequest($this->context . '.filter.fulfilment', 'filter_fulfilment', '');
        $this->setState('filter.fulfilment', $fulfilment);

        $access = $this->getUserStateFromRequest($this->context . '.filter.access', 'filter_access');
        $this->setState('filter.access', $access);

        // List state information.
        parent::populateState($ordering, $direction);
    }

    /**
     * Method to get a store id based on model configuration state.
     *
     * This is necessary because the model is used by the component and
     * different modules that might need different sets of data or different
     * ordering requirements.
     *
     * @param   string  $id  A prefix for the store id.
     *
     *
     * @since   1.0.0
     */
    protected function getStoreId($id = '')
    {
        // Compile the store id.
        $id .= ':' . $this->getState('filter.search');
        $id .= ':' . $this->getState('filter.published');
        $id .= ':' . $this->getState('filter.payment_status');
        $id .= ':' . $this->getState('filter.fulfilment');
        $id .= ':' . $this->getState('filter.access');

        return parent::getStoreId($id);
    }

    /**
     * Method to create a query for a list of items.
     *
     * @return  DatabaseQuery
     *
     * @since  1.0.0
     */
    protected function getListQuery()
    {
        // Create a new query object.
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);
        $user  = $this->getCurrentUser();

        // Select the required fields from the table.
        $query->select(
            $this->getState(
                'list.select',
                'a.*'
            )
        );
        $query->from($db->quoteName('#__easystore_orders', 'a'));

        // Join over the users for the checked out user.
        $query->select($db->quoteName('uc.name', 'editor'))
            ->join('LEFT', $db->quoteName('#__users', 'uc'), $db->quoteName('uc.id') . ' = ' . $db->quoteName('a.checked_out'));

        // Join over the users for the author.
        $query->select([$db->quoteName('ua.name', 'customer_name'),$db->quoteName('ug.title', 'access_title')])
            ->join('LEFT', $db->quoteName('#__easystore_users', 'eu'), $db->quoteName('eu.id') . ' = ' . $db->quoteName('a.customer_id'))
            ->join('LEFT', $db->quoteName('#__users', 'ua'), $db->quoteName('ua.id') . ' = ' . $db->quoteName('eu.user_id'))
            ->join('LEFT', $db->quoteName('#__viewlevels', 'ug'), $db->quoteName('ug.id') . ' = ' . $db->quoteName('a.access'));

        // Filter by access level.
        if ($access = (int) $this->getState('filter.access')) {
            $query->where($db->quoteName('a.access') . ' = :access')
                ->bind(':access', $access, ParameterType::INTEGER);
        }

        // Filter by published state
        $published = (string) $this->getState('filter.published');

        if (is_numeric($published)) {
            $published = (int) $published;
            $query->where($db->quoteName('a.published') . ' = :published')
                ->bind(':published', $published, ParameterType::INTEGER);
        } elseif ($published === '') {
            $query->whereIn($db->quoteName('a.published'), [0, 1]);
        }

        // Filter by payment status
        $paymentStatus = (string) $this->getState('filter.payment_status');
        if (!empty($paymentStatus)) {
            $query->where($db->quoteName('a.payment_status') . ' = :payment_status')
                ->bind(':payment_status', $paymentStatus, ParameterType::STRING);
        }

        // Filter by fulfilment
        $fulfilment = (string) $this->getState('filter.fulfilment');
        if (!empty($fulfilment)) {
            $query->where($db->quoteName('a.fulfilment') . ' = :fulfilment')
                ->bind(':fulfilment', $fulfilment, ParameterType::STRING);
        }

        // Filter by search in title
        $search = $this->getState('filter.search');

        if (!empty($search)) {
            if (stripos($search, 'id:') === 0) {
                $ids = substr($search, 3);
                $query->where($db->quoteName('a.id') . ' = :id')
                    ->bind(':id', $ids, ParameterType::STRING);
            } else {
                $search = '%' . str_replace(' ', '%', trim($search)) . '%';
                $query->where('(' . $db->quoteName('a.fulfilment') . ' LIKE :search1 OR ' . $db->quoteName('a.payment_status') . ' LIKE :search2 OR ' . $db->quoteName('a.order_status') . ' LIKE :search3 OR ' . $db->quoteName('ua.name') . 'LIKE :search4)')
                    ->bind([':search1', ':search2', ':search3', ':search4'], $search);
            }
        }

        // Add the list ordering clause
        $listOrdering = $this->getState('list.ordering', 'a.ordering');
        $listDirn     = $db->escape($this->getState('list.direction', 'ASC'));

        $query->order($db->escape($listOrdering) . ' ' . $listDirn);

        return $query;
    }

    /**
     * Method to get an array of data items.
     *
     * @return  mixed  An array of data items on success, false on failure.
     *
     * @since   1.0.0
     */
    public function getItems()
    {
        $items = parent::getItems();

        foreach ($items as $item) {
            $db    = $this->getDatabase();
            $query = $db->getQuery(true);
            // @todo we will update this query for admin discount
            $subquery = $db->getQuery(true);
            $subquery->select($db->quoteName(['order_id', 'discount_type', 'discount_value', 'price', 'quantity']))
            ->select('SUM(' . $db->quoteName('price') . ' * ' . $db->quoteName('quantity') . ') AS sub_total')
            ->from($db->quoteName('#__easystore_order_product_map'))
            ->where($db->quoteName('order_id') . ' = ' . $item->id)
            ->group([$db->quoteName('order_id'), $db->quoteName('discount_type'), $db->quoteName('discount_value'), $db->quoteName('price'), $db->quoteName('quantity')]);

            // @todo we will update this query for admin discount
            $query->select($db->quoteName(['order_id', 'discount_type', 'discount_value', 'price', 'quantity', 'sub_total']))
            ->select('
                CASE WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                    CASE 
                        WHEN ' . $db->quoteName('discount_type') . ' = ' . $db->quote('percent') . ' THEN 
                        ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                        ELSE ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                    END 
                ELSE
                    0.00
                END AS discounted_price,

                CASE
                    WHEN (
                        CASE
                            WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                                CASE
                                    WHEN ' . $db->quoteName('discount_type') . ' = "percent" THEN
                                       ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                                    ELSE
                                       ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                                END
                            ELSE
                                0.00
                        END
                    ) = 0.00 THEN 0.00
                    ELSE
                    ' . $db->quoteName('price') . ' - (
                            CASE
                                WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                                    CASE
                                        WHEN ' . $db->quoteName('discount_type') . ' = "percent" THEN
                                           ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                                        ELSE
                                           ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                                    END
                                ELSE
                                    0.00
                            END
                        )
                END  AS special_discounted_amount
                ')
            ->from('(' . $subquery . ') AS subquery');


            // Execute the query
            $db->setQuery($query);

            $orderedProducts                 = $db->loadObjectList();
            $item->special_discounted_amount = 0.00;
            $item->discount_amount           = 0.00;
            $item->discounted_sub_total      = 0.00;
            $item->coupon_discount           = 0.00;
            $item->sub_total                 = 0.00;
            $item->shipping_cost             = 0.00;

            if (!empty($orderedProducts)) {
                foreach ($orderedProducts as &$product) {
                    $item->sub_total += floatval($product->sub_total);

                    if (isset($product->discounted_sub_total) && floatval($product->discounted_sub_total) > 0) {
                        $item->discounted_sub_total += floatval($product->discounted_sub_total);
                    }

                    if (!is_null(floatval($product->special_discounted_amount))) {
                        $item->special_discounted_amount += floatval($product->special_discounted_amount);
                    }

                    $item->sub_total -= $item->special_discounted_amount;
                }

                unset($product);
            }

            if (!empty($item->coupon_code)) {
                if (!is_null($item->coupon_amount) && floatval($item->coupon_amount) > 0) {
                    $item->coupon_discount = EasyStoreHelper::calculateDiscountValue($item->coupon_type, $item->coupon_amount, $item->sub_total);
                }
            }

            if (!empty($item->special_discounted_amount)) {
                $item->total_price  = floatval($item->sub_total - $item->special_discounted_amount);
            }

            if (floatval($item->discount_value) > 0) {
                $item->discount_amount = EasyStoreHelper::calculateDiscountValue($item->discount_type, $item->discount_value, $item->sub_total);
            }

            if (!empty($item->shipping) && is_string($item->shipping)) {
                // Convert the shipping string to an object
                $shippingData = json_decode($item->shipping);

                if (isset($shippingData->offerFreeShipping) && $shippingData->offerFreeShipping) {
                    // Check if there's an offer on a specific amount
                    $offerOnAmount = (float) ($shippingData->offerOnAmount ?? null);

                    if ($item->sub_total > $offerOnAmount) {
                        // Apply free shipping if the subtotal is above the offer amount
                        $shippingData->rate = 0;
                    }
                }

                if (isset($shippingData->rate)) {
                    // Format the shipping rate with currency
                    $shippingData->rate_with_currency = EasyStoreHelper::formatCurrency($shippingData->rate);
                }

                // Update the shipping cost with the calculated rate or default to 0
                $item->shipping_cost = $shippingData->rate ?? 0;
            }

            $item->total_price = $item->sub_total + $item->shipping_cost + $item->sale_tax - $item->coupon_discount - $item->discount_amount;
        }

        return $items;
    }
}
